/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;

/**
 * Tests for cancel a workflow run.
 */
public class TestWorkflowRunCanceling extends JobTaskProcessingTestBase {

  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /** basic test for cancel a workflow run in a job run that is in RUNNING state. */
  public void test() throws Exception {
    final String jobName = "testCancelWorkflowRun";
    final String workerParameterValue = "test";
    final String tempStoreName = "tempstore";
    final AnyMap parameters = createJobParameters(workerParameterValue, tempStoreName, tempStoreName);
    addJob(jobName, "testWorkflow", parameters);
    _objectStoreService.ensureStore(tempStoreName);
    final String jobRunId = startJob(jobName);

    // start workflow run 1    
    final Task task_1_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects1 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_1_1 : task_1_1.getOutputBulks().values()) {
      transientObjects1.addAll(bulks_1_1);
    }
    final Task task_1_2 = getSingleNextTask(task_1_1, processInitialTask(task_1_1, tempStoreName), WORKER_2);
    final ResultDescription result1 = processWorker2Task(task_1_2, workerParameterValue, tempStoreName);
    final String wfRun1 = task_1_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // start workflow run 2
    final Task task_2_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects2 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_2_1 : task_2_1.getOutputBulks().values()) {
      transientObjects2.addAll(bulks_2_1);
    }
    final Task task_2_2 = getSingleNextTask(task_2_1, processInitialTask(task_2_1, tempStoreName), WORKER_2);
    final ResultDescription result2 = processWorker2Task(task_2_2, workerParameterValue, tempStoreName);
    final String wfRun2 = task_2_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // cancel workflow run 1
    _jobRunEngine.cancelWorkflowRun(jobName, jobRunId, wfRun1);

    assertObjectsDeleted(transientObjects1);
    try {
      getSingleNextTask(task_1_2, result1, WORKER_3);
      fail("should not work");
    } catch (final IllegalJobStateException ex) {
      ; // OK
    }

    // but workflow run 2 is still alive
    assertObjectsExist(transientObjects2);
    Task task_2_3 = getSingleNextTask(task_2_2, result2, WORKER_3);
    assertNotNull(task_2_3);
    assertEquals(wfRun2, task_2_3.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID));
    createBulk(task_2_3, "output", "result");
    finishFinalTask(task_2_3, successResult());

    _jobRunEngine.finishJob(jobName, jobRunId);
    final AnyMap jobRunData = assertJobRunSucceeded(jobName, jobRunId);
    assertWorkflowCounters(2, 0, 1, 0, 1, jobRunData);
    assertTaskCounters(5, 4, 0, 0, 0, 0, 1, jobRunData);
  }

  /** cancel a workflow run for a job run which is in FINISHING state. */
  public void testFinishingJobRun() throws Exception {
    final String jobName = "testFinishingJobRun";
    final String workerParameterValue = "test";
    final String tempStoreName = "tempstore";
    final AnyMap parameters = createJobParameters(workerParameterValue, tempStoreName, tempStoreName);
    addJob(jobName, "testWorkflow", parameters);
    _objectStoreService.ensureStore(tempStoreName);
    final String jobRunId = startJob(jobName);

    // start workflow run 1    
    final Task task_1_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects1 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_1_1 : task_1_1.getOutputBulks().values()) {
      transientObjects1.addAll(bulks_1_1);
    }
    final Task task_1_2 = getSingleNextTask(task_1_1, processInitialTask(task_1_1, tempStoreName), WORKER_2);
    final ResultDescription result1 = processWorker2Task(task_1_2, workerParameterValue, tempStoreName);
    final String wfRun1 = task_1_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // start workflow run 2
    final Task task_2_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects2 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_2_1 : task_2_1.getOutputBulks().values()) {
      transientObjects2.addAll(bulks_2_1);
    }
    final Task task_2_2 = getSingleNextTask(task_2_1, processInitialTask(task_2_1, tempStoreName), WORKER_2);
    final ResultDescription result2 = processWorker2Task(task_2_2, workerParameterValue, tempStoreName);
    final String wfRun2 = task_2_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // finish job run
    _jobRunEngine.finishJob(jobName, jobRunId);

    // cancel workflow run 1
    _jobRunEngine.cancelWorkflowRun(jobName, jobRunId, wfRun1);

    assertObjectsDeleted(transientObjects1);
    try {
      getSingleNextTask(task_1_2, result1, WORKER_3);
      fail("should not work");
    } catch (final IllegalJobStateException ex) {
      ; // OK
    }

    // but workflow run 2 is still alive
    assertObjectsExist(transientObjects2);
    Task task_2_3 = getSingleNextTask(task_2_2, result2, WORKER_3);
    assertNotNull(task_2_3);
    assertEquals(wfRun2, task_2_3.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID));
    createBulk(task_2_3, "output", "result");
    finishFinalTask(task_2_3, successResult());

    final AnyMap jobRunData = assertJobRunSucceeded(jobName, jobRunId);
    assertWorkflowCounters(2, 0, 1, 0, 1, jobRunData);
    assertTaskCounters(5, 4, 0, 0, 0, 0, 1, jobRunData);
  }

  /** cancel a workflow run which is the only workflow run in the job run. */
  public void testOnlyOneWorkflowRun() throws Exception {
    final String jobName = "testOnlyOneWorkflowRun";
    final String workerParameterValue = "test";
    final String tempStoreName = "tempstore";
    final AnyMap parameters = createJobParameters(workerParameterValue, tempStoreName, tempStoreName);
    addJob(jobName, "testWorkflow", parameters);
    _objectStoreService.ensureStore(tempStoreName);
    final String jobRunId = startJob(jobName);

    // start workflow run 1    
    final Task task_1_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects1 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_1_1 : task_1_1.getOutputBulks().values()) {
      transientObjects1.addAll(bulks_1_1);
    }
    final Task task_1_2 = getSingleNextTask(task_1_1, processInitialTask(task_1_1, tempStoreName), WORKER_2);
    final ResultDescription result1 = processWorker2Task(task_1_2, workerParameterValue, tempStoreName);
    final String wfRun1 = task_1_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // cancel workflow run 1
    _jobRunEngine.cancelWorkflowRun(jobName, jobRunId, wfRun1);

    assertObjectsDeleted(transientObjects1);
    try {
      getSingleNextTask(task_1_2, result1, WORKER_3);
      fail("should not work");
    } catch (final IllegalJobStateException ex) {
      ; // OK
    }

    _jobRunEngine.finishJob(jobName, jobRunId);
    // the job run is FAILED because there is no successful workflow run
    final AnyMap jobRunData = assertJobRunFailed(jobName, jobRunId);
    assertWorkflowCounters(1, 0, 0, 0, 1, jobRunData);
    assertTaskCounters(2, 1, 0, 0, 0, 0, 1, jobRunData);
  }

  /** cancel a workflow run in a runOnce job. */
  public void testRunOnceJob() throws Exception {
    final String jobName = "testRunOnceJob";
    final String workerParameterValue = "test";
    final String tempStoreName = "temp";
    final AnyMap parameters = createJobParameters(workerParameterValue, tempStoreName, tempStoreName);
    addJob(jobName, "testWorkflowWithInput", parameters);

    final int noOfObjects = 3;
    _objectStoreService.ensureStore(tempStoreName);
    final String bucketName = "insertBucket";
    for (int i = 0; i < noOfObjects; i++) {
      _objectStoreService.putObject(tempStoreName, bucketName + "/objectnumber" + i, ("" + i).getBytes());
    }

    final String jobRunId = startJob(jobName, JobRunMode.RUNONCE);

    // cancel workflow run - should have id "1"
    _jobRunEngine.cancelWorkflowRun(jobName, jobRunId, "1");

    waitForJobRunCompleted(jobName, jobRunId, 10000);
    // the job run is FAILED because there is no successful workflow run
    final AnyMap jobRunData = assertJobRunFailed(jobName, jobRunId);
    assertWorkflowCounters(1, 0, 0, 0, 1, jobRunData);
    assertTaskCounters(noOfObjects, 0, 0, 0, 0, 0, noOfObjects, jobRunData);
  }

  /** cancel a workflow run for non-exisiting job run. */
  public void testJobRunDoesNotExist() throws Exception {
    final String jobName = "testJobRunDoesNotExist";
    final String workerParameterValue = "test";
    final String storeName = "test";
    final String tempStoreName = "temp";
    final AnyMap parameters = createJobParameters(workerParameterValue, storeName, tempStoreName);
    addJob(jobName, "testCancelWorkflow", parameters);
    try {
      _jobRunEngine.cancelWorkflowRun(jobName, "nonExistingJobRunId", "1");
    } catch (JobManagerException e) {
      assertTrue(e.getMessage().contains("nonExistingJobRunId"));
    }
  }

  /** cancel a non-exisiting workflow run - this should be ignored. */
  public void testWorkflowRunDoesNotExist() throws Exception {
    final String jobName = "testCancelWorkflowRun";
    final String workerParameterValue = "test";
    final String tempStoreName = "tempstore";
    final AnyMap parameters = createJobParameters(workerParameterValue, tempStoreName, tempStoreName);
    addJob(jobName, "testWorkflow", parameters);
    _objectStoreService.ensureStore(tempStoreName);
    final String jobRunId = startJob(jobName);

    // start a workflow run     
    final Task task_1_1 = _jobTaskProcessor.getInitialTask(WORKER_1, jobName);
    final List<BulkInfo> transientObjects1 = new ArrayList<BulkInfo>();
    for (final List<BulkInfo> bulks_1_1 : task_1_1.getOutputBulks().values()) {
      transientObjects1.addAll(bulks_1_1);
    }
    final Task task_1_2 = getSingleNextTask(task_1_1, processInitialTask(task_1_1, tempStoreName), WORKER_2);
    final ResultDescription result1 = processWorker2Task(task_1_2, workerParameterValue, tempStoreName);
    final String wfRun1 = task_1_2.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // cancel non-existing workflow run - should be ignored
    _jobRunEngine.cancelWorkflowRun(jobName, jobRunId, "nonExistingWorkflowRun");

    // ... so we can just finish our work
    assertObjectsExist(transientObjects1);
    Task task_1_3 = getSingleNextTask(task_1_2, result1, WORKER_3);
    assertNotNull(task_1_3);
    assertEquals(wfRun1, task_1_3.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID));
    createBulk(task_1_3, "output", "result");
    finishFinalTask(task_1_3, successResult());

    _jobRunEngine.finishJob(jobName, jobRunId);
    final AnyMap jobRunData = assertJobRunSucceeded(jobName, jobRunId);
    assertWorkflowCounters(1, 0, 1, 0, 0, jobRunData);
    assertTaskCounters(3, 3, 0, 0, 0, 0, 0, jobRunData);
  }

}
