/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test.httphandler;

import java.util.Date;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.httphandler.JobHandler;
import org.eclipse.smila.jobmanager.httphandler.JobsHandler;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test cases for the {@link JobHandler} and {@link JobsHandler}.
 */
public class TestJobHandler extends DeclarativeServiceTestCase {

  private JobHandler _jobHandler;

  private JobsHandler _jobsHandler;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    RequestHandler service = getService(RequestHandler.class, "(uriPattern=/jobmanager/jobs/\\([^/]+\\)/?$)");
    assertTrue(service instanceof JobHandler);
    _jobHandler = (JobHandler) service;
    service = getService(RequestHandler.class, "(uriPattern=/jobmanager/jobs/?$)");
    assertTrue(service instanceof JobsHandler);
    _jobsHandler = (JobsHandler) service;
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * tests adding, getting and deleting a definition via the request handler.
   */
  public void testAddAndGetAndDeleteDefinition() throws Exception {
    final String jobBaseUrlPart = _jobHandler.getRootContextPath() + "/jobmanager/jobs/";
    final String workflowName = "optionalParamWorkflow";
    final String definitionName = "testAddAndGetAndDeleteDefinition";
    final Record definition = DataFactory.DEFAULT.createRecord();
    definition.getMetadata().put("name", definitionName);
    definition.getMetadata().put("workflow", workflowName);
    final AnyMap parameter = DataFactory.DEFAULT.createAnyMap();
    parameter.put("workerParameter", "workerParameter");
    parameter.put("store", "store");
    definition.getMetadata().put("parameters", parameter);
    definition.getMetadata().put("additionalData", "some additional data.");
    // this must be removed when inserted via handler.
    definition.getMetadata().put(DefinitionBase.KEY_READ_ONLY, true);

    // create a definition
    final Date startDate = new Date();
    final AnyMap result = (AnyMap) _jobsHandler.process("POST", jobBaseUrlPart, definition, null);
    assertEquals(definition.getMetadata().get("name").toString(), result.get("name").toString());
    // get the complete definition
    final Record returnDetails = DataFactory.DEFAULT.createRecord();
    returnDetails.getMetadata().put(JobManagerConstants.KEY_RETURN_DETAILS, true);
    AnyMap retrievedDefinition =
      ((AnyMap) _jobHandler.process("GET", jobBaseUrlPart + definitionName, returnDetails, null))
        .getMap(JobHandler.KEY_DEFINITION);
    final Date endDate = new Date();
    // check the timestamp of the result.
    final Date dateFromDefinition = retrievedDefinition.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP);
    assertEquals(result.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP), dateFromDefinition);
    assertTrue(startDate.compareTo(dateFromDefinition) <= 0);
    assertTrue(endDate.compareTo(dateFromDefinition) >= 0);
    // be sure that the readonly flag has been removed from the input any.
    assertFalse(retrievedDefinition.containsKey(DefinitionBase.KEY_READ_ONLY));
    // we must remove timestamp and readOnly flag for comparison...
    final Any timestampAny = definition.getMetadata().get(JobManagerConstants.TIMESTAMP);
    assertEquals(timestampAny.asValue().asDateTime(), dateFromDefinition);
    definition.getMetadata().remove(DefinitionBase.KEY_TIMESTAMP);
    definition.getMetadata().remove(DefinitionBase.KEY_READ_ONLY);
    retrievedDefinition.remove(JobManagerConstants.TIMESTAMP);
    assertEquals(definition.getMetadata(), retrievedDefinition);
    // get only the relevant part
    retrievedDefinition =
      ((AnyMap) _jobHandler.process("GET", jobBaseUrlPart + definitionName, null, null))
        .getMap(JobHandler.KEY_DEFINITION);
    final AnyMap reducedAny = DataFactory.DEFAULT.cloneAnyMap(definition.getMetadata());
    reducedAny.remove("additionalData");
    // re-set the timestamp to be able to check it (we removed it above)
    reducedAny.put(DefinitionBase.KEY_TIMESTAMP, timestampAny);
    assertEquals(reducedAny, retrievedDefinition);
    // remove the workflow
    _jobHandler.process("DELETE", jobBaseUrlPart + definitionName, null, null);

    // check if it really got removed...
    try {
      _jobHandler.process("GET", jobBaseUrlPart + definitionName, null, null);
      fail("ConfigNotFoundException expected.");
    } catch (final ConfigNotFoundException e) {
      ;// ignore
    }
  }

  /**
   * tests the job run section of the GET result and the "returnRuns" parameter
   */
  public void testJobRunPart() throws Exception {
    final String jobBaseUrlPart = _jobHandler.getRootContextPath() + "/jobmanager/jobs/";
    final String workflowName = "optionalParamWorkflow";
    final String definitionName = "testAddAndGetAndDeleteDefinition";
    final Record definition = DataFactory.DEFAULT.createRecord();
    definition.getMetadata().put("name", definitionName);
    definition.getMetadata().put("workflow", workflowName);
    final AnyMap parameter = DataFactory.DEFAULT.createAnyMap();
    parameter.put("workerParameter", "workerParameter");
    parameter.put("store", "store");
    definition.getMetadata().put("parameters", parameter);
    definition.getMetadata().put("additionalData", "some additional data.");
    // this must be removed when inserted via handler.
    definition.getMetadata().put(DefinitionBase.KEY_READ_ONLY, true);
    _jobsHandler.process("POST", jobBaseUrlPart, definition, null);
    final String jobUrl = jobBaseUrlPart + definitionName;
    final Record parameters = DataFactory.DEFAULT.createRecord();
    final AnyMap defaultResult = (AnyMap) _jobHandler.process("GET", jobUrl, parameters, null);
    assertTrue(defaultResult.containsKey("runs"));
    assertTrue(defaultResult.getMap("runs").containsKey("current"));
    assertTrue(defaultResult.getMap("runs").get("current").isMap());
    assertTrue(defaultResult.getMap("runs").containsKey("history"));
    assertTrue(defaultResult.getMap("runs").get("history").isSeq());
    assertTrue(defaultResult.containsKey("definition"));
    assertFalse(defaultResult.getMap("definition").containsKey("additionalData"));
    parameters.getMetadata().put("returnRuns", true);
    final AnyMap resultWithHistory = (AnyMap) _jobHandler.process("GET", jobUrl, parameters, null);
    assertEquals(defaultResult, resultWithHistory);
    parameters.getMetadata().put("returnRuns", false);
    final AnyMap resultWithoutHistory = (AnyMap) _jobHandler.process("GET", jobUrl, parameters, null);
    assertTrue(resultWithoutHistory.containsKey("definition"));
    assertFalse(resultWithoutHistory.getMap("definition").containsKey("additionalData"));
    assertFalse(resultWithoutHistory.containsKey("runs"));
    parameters.getMetadata().put("returnDetails", true);
    final AnyMap resultWithoutHistoryWithDetails = (AnyMap) _jobHandler.process("GET", jobUrl, parameters, null);
    assertTrue(resultWithoutHistoryWithDetails.containsKey("definition"));
    assertTrue(resultWithoutHistoryWithDetails.getMap("definition").containsKey("additionalData"));
    assertFalse(resultWithoutHistoryWithDetails.containsKey("runs"));

  }

  /** check if a job with an invalid name is rejected. */
  public void testAddJobWithInvalidName() throws Exception {
    try {
      final Record job = DataFactory.DEFAULT.createRecord();
      job.getMetadata().put("name", "this/name+is=invalid?");
      _jobsHandler.process("POST", "jobmanager/jobs", job, null);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK
    }
  }
}
