/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.taskmanager.Task;

/**
 * Implements the handling of HTTP requests of a worker to obtain an initial task.
 * 
 * URL pattern: -workerName-/initialTask/-jobName-/
 */
public class InitialTaskHandler extends AJobManagerHandler {

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() != 2) {
      throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern()
        + "'.");
    }
    final String workerName = uriParts.get(0);
    final String jobName = uriParts.get(1);
    final Task task = getJobTaskProcessor().getInitialTask(workerName, jobName);
    if (task != null) {
      return task.toAny();
    }
    return null;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for InvalidValueTypeException to the exception handling of
   * {@link AJobManagerHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof JobManagerException) {
      return ((JobManagerException) ex).isRecoverable() ? HttpStatus.INTERNAL_SERVER_ERROR : HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * return NO_CONTENT if no task was found.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String requestMethod, final String requestUri, final Record inputRecord,
    final Object resultObject) {
    if (resultObject == null) {
      return HttpStatus.NO_CONTENT;
    }
    return super.getSuccessStatus(requestMethod, requestUri, inputRecord, resultObject);
  }

}
