/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.InputOutput;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.utils.collections.MultiValueMap;

/**
 * Implementation of {@link JobRun}.
 */
public class JobRunImpl implements JobRun {

  /** The id of the job run. */
  private final String _jobRunId;

  /** The JobDefinition of the job run. */
  private final JobDefinition _jobDef;

  /** The WorkflowDefinition of the job run's job. */
  private final WorkflowDefinition _workflowDef;

  /** The startAction of the workflow. */
  private WorkflowAction _startAction;

  /** list of all actions (including start action) for access via position. */
  private final List<WorkflowAction> _actions = new ArrayList<WorkflowAction>();

  /** bucket name -> workflow actions having this bucket as input in this JobRun. */
  private final MultiValueMap<String, WorkflowAction> _inputBucketToAction =
    new MultiValueMap<String, WorkflowAction>();

  /** bucket name -> workflow actions having this bucket as output in this JobRun. */
  private final MultiValueMap<String, WorkflowAction> _outputBucketToAction =
    new MultiValueMap<String, WorkflowAction>();

  /** bucket name -> bucket instance (for this JobRun). */
  private final Map<String, Bucket> _buckets = new HashMap<String, Bucket>();

  /** WorkflowAction -> input slot name -> bucket. */
  private final Map<WorkflowAction, Map<String, Bucket>> _actionInputBuckets =
    new HashMap<WorkflowAction, Map<String, Bucket>>();

  /** WorkflowAction -> output slot name -> bucket. */
  private final Map<WorkflowAction, Map<String, Bucket>> _actionOutputBuckets =
    new HashMap<WorkflowAction, Map<String, Bucket>>();

  /** A map of the workers (key: worker-name, value: WorkerDefinition) for this job run. */
  private final Map<String, WorkerDefinition> _workers = new HashMap<String, WorkerDefinition>();

  /** A map of the merged parameters (job, workflow) (key: param-name, value: ValueExpression) for this job run. */
  private final AnyMap _parameters = DataFactory.DEFAULT.createAnyMap();

  /** A map with bucket names as key and bucket definition as value. */
  private final Map<String, BucketDefinition> _bucketDefinitions = new HashMap<String, BucketDefinition>();

  /** actions that have workers with mode "barrier" set. */
  private final ArrayList<WorkflowAction> _barrierActions = new ArrayList<>();

  /** Barrier actions -> preceding barrier actions. */
  private final MultiValueMap<WorkflowAction, WorkflowAction> _barrierDependencies = new MultiValueMap<>();

  /** Actions -> Barrier actions they contribute to. */
  private final MultiValueMap<WorkflowAction, WorkflowAction> _actionsToBarriers = new MultiValueMap<>();

  /**
   * Constructs the job run data for a job run.
   * 
   * @param runId
   *          the id of the run.
   * @param jobName
   *          The job name.
   * @param runStorage
   *          The runStorage.
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @throws Exception
   *           An exception if something goes wrong
   */
  public JobRunImpl(final String runId, final String jobName, final RunStorage runStorage,
    final DefinitionPersistence definitions) throws Exception {
    super();
    final AnyMap jobRunData = runStorage.getJobRunData(jobName, true);
    final AnyMap jobDefinitionMap = jobRunData.getMap(JobManagerConstants.DATA_JOB_RUN_JOB_DEF);
    final AnyMap workflowDefinitionMap = jobRunData.getMap(JobManagerConstants.DATA_JOB_RUN_WORKFLOW_DEF);
    final AnySeq bucketsSeq = jobRunData.getSeq(JobManagerConstants.DATA_JOB_RUN_BUCKET_DEFS);
    if (bucketsSeq != null && bucketsSeq.size() > 0) {
      final Iterator<Any> bucketsIterator = bucketsSeq.iterator();
      while (bucketsIterator.hasNext()) {
        final AnyMap bucketsAny = (AnyMap) bucketsIterator.next();
        _bucketDefinitions.put(bucketsAny.getStringValue(BucketDefinition.KEY_NAME), new BucketDefinition(
          bucketsAny));
      }
    }
    _jobRunId = runId;
    _jobDef = new JobDefinition(jobDefinitionMap);
    _workflowDef = new WorkflowDefinition(workflowDefinitionMap);
    compileWorkflow(definitions);
  }

  /**
   * Compiles the workflow, i.e. merges the parameters, loads the WorkerDefinitions, creates Bucket informations with
   * the merged parameters.
   * 
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileWorkflow(final DefinitionPersistence definitions) throws PersistenceException {
    if (_jobDef.getParameters() != null) {
      _parameters.putAll(_jobDef.getParameters());
    }
    if (_workflowDef.getParameters() != null) {
      _parameters.putAll(_workflowDef.getParameters());
    }
    _startAction = _workflowDef.getStartAction();
    _actions.add(_startAction);
    compileWorkflowAction(_startAction, definitions);
    if (_workflowDef.getActions() != null) {
      for (final WorkflowAction action : _workflowDef.getActions()) {
        _actions.add(action);
        compileWorkflowAction(action, definitions);
        if (_workers.get(action.getWorker()).getModes().contains(Mode.BARRIER)) {
          _barrierActions.add(action);
        }
      }
      compileBarriers();
    }
  }

  /**
   * Compiles a single WorkflowAction (i.e. loads the WorkerDefinitions, creates Bucket informations with the merged
   * parameters).
   * 
   * If the isStartAction flag is set to true and the actions worker has 'bulkSource' mode, the worker is marked as
   * being able to start initial tasks.
   * 
   * @param action
   *          The WorkflowAction to compile.
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileWorkflowAction(final WorkflowAction action, final DefinitionPersistence definitions)
    throws PersistenceException {
    final String workerName = action.getWorker();
    final WorkerDefinition workerDef = getWorkerDefinition(definitions, workerName);
    _actionInputBuckets.put(action, new HashMap<String, Bucket>());
    _actionOutputBuckets.put(action, new HashMap<String, Bucket>());

    // input
    final Map<String, String> actionInputSlotsDefinition = action.getInput();
    final Collection<? extends InputOutput<?>> workerInputSlots = workerDef.getInput();
    compileActionSlotConfigurations(action, definitions, actionInputSlotsDefinition, workerInputSlots, true);

    // output
    final Map<String, String> actionOutputSlotsDefinition = action.getOutput();
    final Collection<? extends InputOutput<?>> workerOutputSlots = workerDef.getOutput();
    compileActionSlotConfigurations(action, definitions, actionOutputSlotsDefinition, workerOutputSlots, false);
  }

  /**
   * Compiles the slots (Input or Output) of a single WorkflowAction (i.e. loads the WorkerDefinitions, creates Bucket
   * informations with the merged parameters).
   * 
   * @param action
   *          The WorkflowAction of the slots.
   * @param definitions
   *          The definition persistence where the Bucket and Data Object Type definitions are stored.
   * @param actionSlots
   *          A map of the actions slot to bucket definition
   * @param workerSlots
   *          The slots of a worker
   * @param isInputSlots
   *          true if we are handling input slots
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileActionSlotConfigurations(final WorkflowAction action,
    final DefinitionPersistence definitions, final Map<String, String> actionSlots,
    final Collection<? extends InputOutput<?>> workerSlots, final boolean isInputSlots) throws PersistenceException {
    if (actionSlots != null) {
      for (final Map.Entry<String, String> slot : actionSlots.entrySet()) {
        final String slotName = slot.getKey();
        final String bucketName = slot.getValue();
        Bucket bucket = _buckets.get(bucketName);
        if (bucket == null) {
          BucketDefinition bucketDef = _bucketDefinitions.get(bucketName);
          boolean isPersistent = true;
          if (bucketDef == null) {
            final String slotDataObjectType = getSlotType(slotName, workerSlots);
            isPersistent = false;
            bucketDef = new BucketDefinition(bucketName, slotDataObjectType);
          }
          final DataObjectTypeDefinition dot = definitions.getDataObjectType(bucketDef.getDataObjectType());
          // Use compiled parameters. Job-Parameters and Workflow-Parameters, but not WorkflowActions, since
          // workflow-action parameters must not influence buckets
          bucket = new Bucket(bucketDef, dot, isPersistent, _parameters);
          _buckets.put(bucketName, bucket);
        }
        if (isInputSlots) {
          _actionInputBuckets.get(action).put(slotName, bucket);
          _inputBucketToAction.add(bucketName, action);
        } else {
          _actionOutputBuckets.get(action).put(slotName, bucket);
          _outputBucketToAction.add(bucketName, action);
        }
      }
    }
  }

  /**
   * Returns the data object type of a slot with the given name.
   * 
   * @param slotName
   *          the name of the slot
   * @param slots
   *          a collection of Input or Output elements (describing the slots of the workers)
   * @return The data object type of a slot with the given name.
   */
  private String getSlotType(final String slotName, final Collection<? extends InputOutput<?>> slots) {
    if (slots != null) {
      for (final InputOutput<?> iterableElement : slots) {
        if (iterableElement.getName().equals(slotName)) {
          return iterableElement.getType();
        }
      }
    }
    return null;
  }

  /**
   * Gets a WorkerDefinition from a given DefinitionPersistence instance.
   * 
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @param workerName
   *          The name of the WorkerDefinition to load.
   * @return the WorkerDefinition for workerName.
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private WorkerDefinition getWorkerDefinition(final DefinitionPersistence definitions, final String workerName)
    throws PersistenceException {
    WorkerDefinition workerDef = _workers.get(workerName);
    if (workerDef == null) {
      workerDef = definitions.getWorker(workerName);
      _workers.put(workerName, workerDef);
    }
    return workerDef;
  }

  private void compileBarriers() {
    if (!_barrierActions.isEmpty()) {
      for (final WorkflowAction barrier : _barrierActions) {
        findActionsForBarrier(barrier, barrier);
      }
    }
  }

  private void findActionsForBarrier(final WorkflowAction action, final WorkflowAction barrier) {
    final Map<String, Bucket> inputBucketsForAction = getInputBucketsForAction(action);
    if (inputBucketsForAction != null) {
      for (final Bucket bucket : inputBucketsForAction.values()) {
        final Collection<WorkflowAction> inputActions =
          _outputBucketToAction.get(bucket.getBucketDefinition().getName());
        if (inputActions != null) {
          for (final WorkflowAction inputAction : inputActions) {
            if (!_actionsToBarriers.containsKey(inputAction)
              || !_actionsToBarriers.get(inputAction).contains(barrier)) { // check for action cycles
              _actionsToBarriers.add(inputAction, barrier);
              if (inputAction != barrier) { // check for barrier cycles
                if (_barrierActions.contains(inputAction)) {
                  _barrierDependencies.add(barrier, inputAction);
                }
                findActionsForBarrier(inputAction, barrier);
              }
            }
          }
        }
      }
    }
  }

  @Override
  public String getJobName() {
    return _jobDef.getName();
  }

  @Override
  public String getJobRunId() {
    return _jobRunId;
  }

  @Override
  public WorkflowAction getStartAction() {
    return _startAction;
  }

  @Override
  public WorkflowAction getAction(final int position) {
    return _actions.get(position);
  }

  @Override
  public AnyMap getParameters(final WorkflowAction action) {
    return TaskParameterUtils.mergeAndEvaluateParameters(_jobDef.getParameters(), _workflowDef.getParameters(),
      action.getParameters(), action.getWorker());
  }

  @Override
  public JobDefinition getJobDefinition() {
    return _jobDef;
  }

  @Override
  public WorkflowDefinition getWorkflowDefinition() {
    return _workflowDef;
  }

  @Override
  public Map<String, Bucket> getInputBucketsForAction(final WorkflowAction action) {
    return _actionInputBuckets.get(action);
  }

  @Override
  public Map<String, Bucket> getOutputBucketsForAction(final WorkflowAction action) {
    return _actionOutputBuckets.get(action);
  }

  @Override
  public Collection<Bucket> getBuckets() {
    return _buckets.values();
  }

  @Override
  public Bucket getBucket(final String bucketName) {
    return _buckets.get(bucketName);
  }

  @Override
  public boolean isTriggeredBy(final Bucket triggeringBucket) {
    if (_startAction.getInput() != null) {
      for (final String inputBucketName : _startAction.getInput().values()) {
        final Bucket inputBucket = getBucket(inputBucketName);
        if (inputBucket.isPersistent() && triggeringBucket.getBucketId().equals(inputBucket.getBucketId())) {
          return true;
        }
      }
    }
    return false;
  }

  @Override
  public Collection<WorkflowAction> getTriggeredActionsForBucket(final String bucketName) {
    return _inputBucketToAction.get(bucketName);
  }

  @Override
  public Collection<WorkflowAction> getActionsWritingToBucket(final String bucketName) {
    return _outputBucketToAction.get(bucketName);
  }

  @Override
  public boolean hasBarriers() {
    return !_barrierActions.isEmpty();
  }

  @Override
  public Collection<WorkflowAction> getBarrierActions() {
    return _barrierActions;
  }

  @Override
  public Collection<WorkflowAction> getPrecedingBarriers(final WorkflowAction barrier) {
    return _barrierDependencies.get(barrier);
  }

  @Override
  public Collection<WorkflowAction> getBarriersForAction(final WorkflowAction action) {
    return _actionsToBarriers.get(action);
  }

  @Override
  public Collection<Bucket> getTriggerBuckets() {
    final Collection<Bucket> startBuckets = new ArrayList<Bucket>();
    if (_startAction.getInput() != null) {
      for (final String inputBucketName : _startAction.getInput().values()) {
        final Bucket inputBucket = getBucket(inputBucketName);
        if (inputBucket.isPersistent()) {
          startBuckets.add(inputBucket);
        }
      }
    }
    return startBuckets;
  }

}
