/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.processing.bpel.test;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.utils.config.ConfigUtils;

/**
 * Base class for WorkflowProcessor tests.
 * 
 * @author jschumacher
 * 
 */
public abstract class AWorkflowProcessorTest extends DeclarativeServiceTestCase {
  /**
   * WorkflowProcessor instance to test.
   */
  private WorkflowProcessor _processor;

  /**
   * Blackboard service to use in test.
   */
  private Blackboard _blackboard;

  /** The log. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Check if WorkflowProcessor service is active. Wait up to 30 seconds for start. Fail, if no service is starting.
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _processor = getService(WorkflowProcessor.class);
    assertNotNull("no WorkflowProcessor service found.", _processor);
    final BlackboardFactory factory = getService(BlackboardFactory.class);
    assertNotNull("no BlackboardFactory service found.", factory);
    _blackboard = factory.createPersistingBlackboard();
    assertNotNull("no Blackboard created", _blackboard);
    assertTrue(getPipelineName() + " not active", getProcessor().getWorkflowNames().contains(getPipelineName()));
    _log.info("*** TESTING PIPELINE " + getPipelineName() + " ***");
  }

  /**
   * @return the processor
   */
  public WorkflowProcessor getProcessor() {
    return _processor;
  }

  /**
   * @return the blackboard
   */
  public Blackboard getBlackboard() {
    return _blackboard;
  }

  /**
   * @return the blackboard
   */
  public DataFactory getFactory() {
    return _blackboard.getDataFactory();
  }

  /**
   * @return name of test pipeline
   */
  protected abstract String getPipelineName();

  /**
   * create a new record on the blackboard.
   * 
   * @param source
   *          source value of ID
   * @param key
   *          key value of ID
   * @return id of created record.
   * @throws BlackboardAccessException
   */
  protected String createBlackboardRecord(final String source, final String key) throws BlackboardAccessException {
    final String id = source + ":" + key;
    _blackboard.unloadRecord(id);
    _blackboard.getRecord(id, Get.NEW);
    return id;
  }

  /**
   * read a custom pipeline, check that definition contains the expected pipeline name and BPEl code as well as a
   * timestamp, but no readonly flag.
   */
  protected static void assertWorkflowDefinition(final String pipelineName, final String bpel,
    final AnyMap definition) {
    assertNotNull(definition);
    assertEquals(pipelineName, definition.getStringValue(WorkflowProcessor.WORKFLOW_NAME));
    assertEquals(bpel, definition.getStringValue(WorkflowProcessor.WORKFLOW_DEFINITION));
    assertNotNull(definition.getDateTimeValue(WorkflowProcessor.WORKFLOW_TIMESTAMP));
    assertFalse(definition.containsKey(WorkflowProcessor.WORKFLOW_READONLY));
  }

  /** create definition structure. */
  protected static AnyMap createWorkflowDefinition(final String pipelineName, final String bpel) {
    final AnyMap definition = DataFactory.DEFAULT.createAnyMap();
    definition.put(WorkflowProcessor.WORKFLOW_NAME, pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, bpel);
    return definition;
  }

  /** create new BPEL by reading a configuration file and replacing the name. */
  protected static String copyEchoPipeline(final String pipelineName) throws IOException {
    return copyPipeline("echopipeline.bpel", TestEchoPipeline.PIPELINE_NAME, pipelineName);
  }

  /** create new BPEL by reading a configuration file and replacing the name. */
  protected static String copyLocalHelloWorldPipeline(final String pipelineName) throws IOException {
    return copyPipeline("localhelloworldpipeline.bpel", TestLocalHelloWorldPipeline.PIPELINE_NAME, pipelineName);
  }

  /** create new BPEL by reading a configuration file and replacing the name. */
  protected static String copyHelloWorldPipeline(final String pipelineName) throws IOException {
    return copyPipeline("helloworldpipeline.bpel", TestHelloWorldPipeline.PIPELINE_NAME, pipelineName);
  }

  /** create new BPEL by reading a configuration file and replacing the name. */
  protected static String copySuperPipeline(final String pipelineName) throws IOException {
    return copyPipeline("superpipeline.bpel", TestSuperPipeline.PIPELINE_NAME, pipelineName);
  }

  /** create new BPEL by reading a configuration file and replacing the name. */
  protected static String copyPipeline(final String fileName, final String originalPipelineName,
    final String newPipelineName) throws IOException {
    final String helloBpel = getBpelFile(fileName);
    final String newHelloBpel = helloBpel.replace(originalPipelineName, newPipelineName);
    return newHelloBpel;
  }

  /** read a file from the "pipelines" directory of this bundle's configuration. */
  protected static String getBpelFile(final String filename) throws IOException {
    return ConfigUtils.getConfigContent("org.eclipse.smila.processing.bpel", "pipelines/" + filename);
  }

}
