/**********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.pipelets.test;

import javax.xml.bind.JAXBException;

import org.apache.commons.codec.binary.Base64;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.CodecPipelet;

/**
 * Test cases for {@link CodecPipelet}.
 * 
 * @author aschank
 * 
 */
public class TestCodecPipelet extends APipeletTest {
  /**
   * bundle name for config loading.
   */
  public static final String CONFIG_BUNDLE = "org.eclipse.smila.processing.pipelets";

  /** BASE64 codec. */
  private final Base64 _codec = new Base64();

  /**
   * create and configure CodecPipelet.
   * 
   * @param configuration
   *          {@link AnyMap} with configuration
   * @return configured pipelet.
   * @throws ProcessingException
   *           error configuring pipelet
   */
  public CodecPipelet createPipelet(final AnyMap configuration) throws ProcessingException, JAXBException {
    final CodecPipelet pipelet = new CodecPipelet();
    pipelet.configure(configuration);
    return pipelet;
  }

  /**
   * a very simple first test with one encoded attributes.
   * 
   * @throws Exception
   *           test failed
   */
  public void testDecodeSingleDocumentUsingDefaults() throws Exception {
    // use defaults: empty config
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final CodecPipelet pipelet = createPipelet(config);
    final String id = createBlackboardRecord("source", "key");
    final AnyMap anyMap = getBlackboard().getMetadata(id);
    final String testString = "This is the decoded content.";
    final String testName = "my_name.xyz";
    final String encoded =
      new String(_codec.encode(testString.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    final AnyMap attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded);
    anyMap.add(CodecPipelet.DEFAULT_ATTRIBUTE_NAME, attributesMap);
    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);
    assertTrue(record.hasAttachment(testName));
    final String attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName), CodecPipelet.CHARSET);
    assertEquals(testString, attachmentText);
    // base64 encoded attachments must not be removed
    assertTrue(record.getMetadata().containsKey(CodecPipelet.DEFAULT_ATTRIBUTE_NAME));
  }

  /**
   * Decode multiple encoded attributes.
   * 
   * @throws Exception
   *           test failed
   */
  public void testDecodeMultipleDocumentUsingDefaults() throws Exception {
    // use defaults: empty config
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final CodecPipelet pipelet = createPipelet(config);
    final String id = createBlackboardRecord("source", "key");
    final AnyMap anyMap = getBlackboard().getMetadata(id);
    final String testString1 = "This is the decoded content.";
    final String testString2 = "This is the second decoded content.";
    final String testName1 = "my_name.xyz";
    final String testName2 = "my_name2.xyz";
    final String encoded1 =
      new String(_codec.encode(testString1.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    final String encoded2 =
      new String(_codec.encode(testString2.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    AnyMap attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName1);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded1);
    anyMap.add(CodecPipelet.DEFAULT_ATTRIBUTE_NAME, attributesMap);
    attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName2);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded2);
    anyMap.add(CodecPipelet.DEFAULT_ATTRIBUTE_NAME, attributesMap);
    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);
    assertTrue(record.hasAttachment(testName1));
    assertTrue(record.hasAttachment(testName2));
    String attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName1), CodecPipelet.CHARSET);
    assertEquals(testString1, attachmentText);
    attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName2), CodecPipelet.CHARSET);
    assertEquals(testString2, attachmentText);
    // base64 encoded attachments must not be removed
    assertTrue(record.getMetadata().containsKey(CodecPipelet.DEFAULT_ATTRIBUTE_NAME));
    assertEquals(testName1, record.getMetadata().getSeq(CodecPipelet.DEFAULT_ATTRIBUTE_NAME).getMap(0)
      .getStringValue(CodecPipelet.KEY_NAME));
    assertEquals(encoded1, record.getMetadata().getSeq(CodecPipelet.DEFAULT_ATTRIBUTE_NAME).getMap(0)
      .getStringValue(CodecPipelet.KEY_CONTENT));
    assertEquals(testName2, record.getMetadata().getSeq(CodecPipelet.DEFAULT_ATTRIBUTE_NAME).getMap(1)
      .getStringValue(CodecPipelet.KEY_NAME));
    assertEquals(encoded2, record.getMetadata().getSeq(CodecPipelet.DEFAULT_ATTRIBUTE_NAME).getMap(1)
      .getStringValue(CodecPipelet.KEY_CONTENT));
  }

  /**
   * Decode multiple encoded attributes with embedded configuration.
   * 
   * @throws Exception
   *           test failed
   */
  public void testDecodeMultipleDocumentUsingEmbeddedConfiguration() throws Exception {
    final String attName = "myAtt";
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    config.put(CodecPipelet.PROP_ENCODE, false);
    config.put(CodecPipelet.PROP_ENCODE, false);
    config.put(CodecPipelet.PROP_ATTRIBUTE_NAME, attName);
    config.put(CodecPipelet.PROP_CODEC_NAME, CodecPipelet.DEFAULT_ENCODING);
    // pipelet itself won't be configured
    final CodecPipelet pipelet = createPipelet(DataFactory.DEFAULT.createAnyMap());
    final String id = createBlackboardRecord("source", "key");
    final AnyMap anyMap = getBlackboard().getMetadata(id);
    anyMap.put(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, config);
    final String testString1 = "This is the decoded content.";
    final String testString2 = "This is the second decoded content.";
    final String testName1 = "my_name.xyz";
    final String testName2 = "my_name2.xyz";
    final String encoded1 =
      new String(_codec.encode(testString1.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    final String encoded2 =
      new String(_codec.encode(testString2.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    AnyMap attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName1);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded1);
    anyMap.add(attName, attributesMap);
    attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName2);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded2);
    anyMap.add(attName, attributesMap);
    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);
    assertTrue(record.hasAttachment(testName1));
    assertTrue(record.hasAttachment(testName2));
    String attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName1), CodecPipelet.CHARSET);
    assertEquals(testString1, attachmentText);
    attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName2), CodecPipelet.CHARSET);
    assertEquals(testString2, attachmentText);
    assertTrue(record.getMetadata().containsKey(attName));
    assertEquals(encoded1, record.getMetadata().getSeq(attName).getMap(0).getStringValue(CodecPipelet.KEY_CONTENT));
    assertEquals(encoded2, record.getMetadata().getSeq(attName).getMap(1).getStringValue(CodecPipelet.KEY_CONTENT));
  }

  /**
   * Decode multiple encoded attributes with embedded configuration.
   * 
   * @throws Exception
   *           test failed
   */
  public void testDecodeMultipleDocumentDeleteInput() throws Exception {
    final String attName = "myAtt";
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    // retain input
    config.put(CodecPipelet.PROP_REMOVE_INPUT, true);
    config.put(CodecPipelet.PROP_ENCODE, false);
    config.put(CodecPipelet.PROP_ATTRIBUTE_NAME, attName);
    config.put(CodecPipelet.PROP_CODEC_NAME, CodecPipelet.DEFAULT_ENCODING);
    // pipelet itself won't be configured
    final CodecPipelet pipelet = createPipelet(DataFactory.DEFAULT.createAnyMap());
    final String id = createBlackboardRecord("source", "key");
    final AnyMap anyMap = getBlackboard().getMetadata(id);
    anyMap.put(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, config);
    final String testString1 = "This is the decoded content.";
    final String testString2 = "This is the second decoded content.";
    final String testName1 = "my_name.xyz";
    final String testName2 = "my_name2.xyz";
    final String encoded1 =
      new String(_codec.encode(testString1.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    final String encoded2 =
      new String(_codec.encode(testString2.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    AnyMap attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName1);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded1);
    anyMap.add(attName, attributesMap);
    attributesMap = DataFactory.DEFAULT.createAnyMap();
    attributesMap.put(CodecPipelet.KEY_NAME, testName2);
    attributesMap.put(CodecPipelet.KEY_CONTENT, encoded2);
    anyMap.add(attName, attributesMap);
    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);
    assertTrue(record.hasAttachment(testName1));
    assertTrue(record.hasAttachment(testName2));
    String attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName1), CodecPipelet.CHARSET);
    assertEquals(testString1, attachmentText);
    attachmentText = new String(_blackboard.getAttachmentAsBytes(id, testName2), CodecPipelet.CHARSET);
    assertEquals(testString2, attachmentText);
    // input must still be available
    assertFalse(record.getMetadata().containsKey(attName));
  }

  /**
   * a very simple encoding test with one attachment.
   * 
   * @throws Exception
   *           test failed
   */
  public void testEncodeSingleDocumentUsingDefaults() throws Exception {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    config.put(CodecPipelet.PROP_ENCODE, true);
    final CodecPipelet pipelet = createPipelet(config);
    final String id = createBlackboardRecord("source", "key");
    final String testString = "This is the decoded content.";
    final String testName = "my_name.xyz";
    final String encoded =
      new String(_codec.encode(testString.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);

    // store the attachment
    _blackboard.setAttachment(id, testName, testString.getBytes(CodecPipelet.CHARSET));

    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);
    // by default: attachment is not removed
    assertTrue(record.hasAttachment(testName));
    assertTrue(record.getMetadata().containsKey(CodecPipelet.DEFAULT_ATTRIBUTE_NAME));
    final AnyMap attachmentMap = record.getMetadata().getSeq(CodecPipelet.DEFAULT_ATTRIBUTE_NAME).getMap(0);
    assertNotNull(attachmentMap);
    final String encodedAttachmentText = attachmentMap.getStringValue(CodecPipelet.KEY_CONTENT);
    assertEquals(testName, attachmentMap.getStringValue(CodecPipelet.KEY_NAME));
    assertNotNull(encodedAttachmentText);
    assertEquals(encoded, encodedAttachmentText);
  }

  /**
   * an encoding test with two attachments.
   * 
   * @throws Exception
   *           test failed
   */
  public void testEncodeMultipleDocumentsRemoveInput() throws Exception {
    final String attName = "xyzAtt";
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    config.put(CodecPipelet.PROP_ENCODE, true);
    config.put(CodecPipelet.PROP_REMOVE_INPUT, true);
    config.put(CodecPipelet.PROP_ATTRIBUTE_NAME, attName);
    final CodecPipelet pipelet = createPipelet(DataFactory.DEFAULT.createAnyMap());
    final String id = createBlackboardRecord("source", "key");
    final AnyMap anyMap = getBlackboard().getMetadata(id);
    anyMap.put(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, config);
    final String testString1 = "This is the decoded content.";
    final String testName1 = "my_name.xyz";
    final String encoded1 =
      new String(_codec.encode(testString1.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    final String testString2 = "This is the second decoded content.";
    final String testName2 = "my_name2.xyz";
    final String encoded2 =
      new String(_codec.encode(testString2.getBytes(CodecPipelet.CHARSET)), CodecPipelet.CHARSET);
    // store the attachments
    _blackboard.setAttachment(id, testName1, testString1.getBytes(CodecPipelet.CHARSET));
    _blackboard.setAttachment(id, testName2, testString2.getBytes(CodecPipelet.CHARSET));

    pipelet.process(getBlackboard(), new String[] { id });
    final Record record = getBlackboard().getRecord(id);

    // attachments are removed.
    assertFalse(record.hasAttachments());

    assertFalse(record.hasAttachment(testName1));
    assertTrue(record.getMetadata().containsKey(attName));
    final AnyMap attachmentMap = record.getMetadata().getSeq(attName).getMap(0);
    assertNotNull(attachmentMap);
    final String encodedAttachmentText = attachmentMap.getStringValue(CodecPipelet.KEY_CONTENT);
    assertEquals(testName1, attachmentMap.getStringValue(CodecPipelet.KEY_NAME));
    assertNotNull(encodedAttachmentText);
    assertEquals(encoded1, encodedAttachmentText);

    assertFalse(record.hasAttachment(testName2));
    final AnyMap attachmentMap2 = record.getMetadata().getSeq(attName).getMap(1);
    assertNotNull(attachmentMap2);
    final String encodedAttachmentText2 = attachmentMap2.getStringValue(CodecPipelet.KEY_CONTENT);
    assertEquals(testName2, attachmentMap2.getStringValue(CodecPipelet.KEY_NAME));
    assertNotNull(encodedAttachmentText2);
    assertEquals(encoded2, encodedAttachmentText2);
  }
}
