/*******************************************************************************
 * Copyright (c) 2013, Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program
 * and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Palmar (Empolis Information Management GmbH) - initial implementation
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets.test;

import javax.xml.bind.JAXBException;

import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.pipelets.DocumentSplitterPipelet;

public class TestDocumentSplitterPipelet extends APipeletTest {

  public static final String PARTS_ATTRIBUTE_NAME = "parts";

  public DocumentSplitterPipelet createPipelet() throws ProcessingException, JAXBException {
    final AnyMap config = getBlackboard().getDataFactory().createAnyMap();
    config.put(DocumentSplitterPipelet.PROP_PARTS_ATTRIBUTE, PARTS_ATTRIBUTE_NAME);
    final DocumentSplitterPipelet pipelet = new DocumentSplitterPipelet();
    pipelet.configure(config);
    return pipelet;
  }

  public void testSplitDocument() throws Exception {
    final DocumentSplitterPipelet pipelet = createPipelet();
    final DataFactory dataFactory = _blackboard.getDataFactory();

    // without document parts
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnyMap expected = (AnyMap) dataFactory.cloneAny(metaData);
      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      assertEquals(expected, result);
    }

    // empty value of document parts attribute
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnyMap expected = (AnyMap) dataFactory.cloneAny(metaData);
      metaData.put(PARTS_ATTRIBUTE_NAME, dataFactory.createAnySeq());
      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      assertEquals(expected, result);
    }

    // one empty document part
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnySeq parts = dataFactory.createAnySeq();
      parts.add(dataFactory.createAnyMap());
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);
      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      final AnyMap expected = dataFactory.createAnyMap();
      setRecordAndDocumentId(expected, recordId, 0);
      assertEquals(expected, result);
    }

    // more empty document parts
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnySeq parts = dataFactory.createAnySeq();
      parts.add(dataFactory.createAnyMap());
      parts.add(dataFactory.createAnyMap());
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);
      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(2, resultIds.length);

      final AnyMap expected = dataFactory.createAnyMap();
      for (int partNo = 0; partNo < 2; ++partNo) {
        final Any result = getBlackboard().getMetadata(resultIds[partNo]);
        setRecordAndDocumentId(expected, recordId, partNo);
        assertEquals(expected, result);
      }
    }

    // one document part with one attribute
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnySeq parts = dataFactory.createAnySeq();
      final AnyMap part = dataFactory.createAnyMap();
      part.put("color", "white");
      parts.add(part);
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);

      final AnyMap expected = (AnyMap) dataFactory.cloneAny(part);
      setRecordAndDocumentId(expected, recordId, 0);

      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      assertEquals(expected, result);
    }

    // one document part with more attributes
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      final AnySeq parts = dataFactory.createAnySeq();
      final AnyMap part = dataFactory.createAnyMap();
      part.put("boolAtt", false);
      part.put("numAtt", 1.2);
      part.put("strAtt", "str");
      final AnySeq seq = dataFactory.createAnySeq();
      seq.add(3);
      part.put("seqAtt", seq);
      final AnyMap map = dataFactory.createAnyMap();
      map.put("key", 4);
      part.put("mapAtt", map);
      parts.add(part);
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);

      final AnyMap expected = (AnyMap) dataFactory.cloneAny(part);
      setRecordAndDocumentId(expected, recordId, 0);

      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      assertEquals(expected, result);
    }

    // global attribute unchanged, another global attribute overwritten, one attribute added
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      metaData.put("shape", "line");
      metaData.put("show", false);
      final AnySeq parts = dataFactory.createAnySeq();
      final AnyMap part = dataFactory.createAnyMap();
      part.put("color", "white");
      part.put("show", true);
      parts.add(part);
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);

      final AnyMap expected = (AnyMap) dataFactory.cloneAny(part);
      expected.put("shape", "line");
      setRecordAndDocumentId(expected, recordId, 0);

      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(1, resultIds.length);
      final Any result = getBlackboard().getMetadata(resultIds[0]);
      assertEquals(expected, result);
    }

    // more document parts
    {
      final String recordId = "abc.pdf";
      final Record record = _blackboard.getRecord(recordId, Get.NEW);
      final AnyMap metaData = record.getMetadata();
      metaData.put("shape", "line");
      metaData.put("show", false);
      final AnySeq parts = dataFactory.createAnySeq();
      AnyMap part = dataFactory.createAnyMap();
      part.put("color", "white");
      part.put("show", true);
      parts.add(part);
      parts.add(dataFactory.createAnyMap());
      part = dataFactory.createAnyMap();
      part.put("length", 2.5);
      parts.add(part);
      metaData.put(PARTS_ATTRIBUTE_NAME, parts);

      final AnySeq expecteds = dataFactory.createAnySeq();
      for (int partNo = 0; partNo < 3; ++partNo) {
        final AnyMap expected = dataFactory.cloneAnyMap(metaData);
        expected.putAll(parts.get(partNo).asMap());
        expected.remove(PARTS_ATTRIBUTE_NAME);
        setRecordAndDocumentId(expected, recordId, partNo);
        expecteds.add(expected);
      }

      final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
      assertEquals(3, resultIds.length);
      for (int partNo = 0; partNo < 3; ++partNo) {
        final Any expected = expecteds.get(partNo);
        final Any result = getBlackboard().getMetadata(resultIds[partNo]);
        assertEquals(expected, result);
      }
    }
  }

  /** tests that attachments are not contained in splitted records. */
  public void testAttachments() throws Exception {
    final DocumentSplitterPipelet pipelet = createPipelet();
    final DataFactory dataFactory = _blackboard.getDataFactory();

    final String recordId = "r1";
    final Record record = _blackboard.getRecord(recordId, Get.NEW);
    final AnyMap metaData = record.getMetadata();

    final AnySeq parts = dataFactory.createAnySeq();
    final AnyMap part1 = dataFactory.createAnyMap();
    part1.put("partAtt", "v1");
    parts.add(part1);
    final AnyMap part2 = dataFactory.createAnyMap();
    part2.put("partAtt", "v2");
    parts.add(part2);
    metaData.put(PARTS_ATTRIBUTE_NAME, parts);

    record.setAttachment("attachment", new byte[10]);

    final String[] resultIds = pipelet.process(getBlackboard(), new String[] { recordId });
    assertEquals(2, resultIds.length);
    final Record result1 = getBlackboard().getRecord(resultIds[0]);
    final Record result2 = getBlackboard().getRecord(resultIds[1]);
    assertFalse(result1.hasAttachments());
    assertFalse(result2.hasAttachments());
  }

  private void setRecordAndDocumentId(final AnyMap record, final String recordId, final int partNo) {
    final String newRecordId = recordId + DocumentSplitterPipelet.SPLIT_ID_SEPARATOR + partNo;
    record.put("_recordid", newRecordId);
    record.put(DocumentSplitterPipelet.DOCUMENT_ID, recordId);
  }
}
