/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. 
 * All rights reserved. This program and the accompanying materials are made available 
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - implementation
 *******************************************************************************/

package org.eclipse.smila.processing.pipelets.test;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.pipelets.FilterPipelet;

/**
 * Test case for the FilterPipelet.
 */
public class TestFilterPipelet extends APipeletTest {

  /**
   * create and configure FilterPipelet instance.
   */
  public FilterPipelet createPipelet() throws ProcessingException {
    final FilterPipelet pipelet = new FilterPipelet();
    final AnyMap configuration = DataFactory.DEFAULT.createAnyMap();
    configuration.put("filterAttribute", "_recordid");
    configuration.put("filterExpression", ".+:[0-5]{1}");
    pipelet.configure(configuration);
    return pipelet;
  }

  /**
   * Test filter records by Id.
   */
  public void testFilterRecords() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 6;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /**
   * Test record specific parameter override.
   */
  public void testFilterRecordsSpecific() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
      if (i == 6) {
        getBlackboard().getMetadata(recordIds[i]).getMap("_parameters", true).put("filterExpression", "filter:.+");
      }
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 7;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /**
   * Test dropping of records with filter attribute not set.
   */
  public void testUnsetFilterAttribute() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
      if (i == 5) {
        getBlackboard().getMetadata(recordIds[i]).getMap("_parameters", true).put("filterAttribute", "unknown");
      }
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 5;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /**
   * Test dropping of records with invalid filter expression.
   */
  public void testInvalidFilterExpression() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
      if (i == 5) {
        getBlackboard().getMetadata(recordIds[i]).getMap("_parameters", true).put("filterExpression", "+");
      }
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 5;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /**
   * Test dropping of records with invalid value type.
   */
  public void testInvalidFilterValueType() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
      if (i == 5) {
        getBlackboard().getMetadata(recordIds[i]).getMap("_parameters", true).put("filterAttribute", "_parameters");
      }
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 5;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /**
   * Test dropping of records with invalid record Ids.
   */
  public void testInvalidRecordId() throws Exception {
    final FilterPipelet pipelet = createPipelet();
    final int noOfRecords = 12;
    final String[] recordIds = new String[noOfRecords];
    for (int i = 0; i < noOfRecords; i++) {
      if (i == 5) {
        recordIds[i] = "unknown";
      } else {
        recordIds[i] = createBlackboardRecord("filter", Integer.toString(i));
      }
    }
    final String[] matchingRecords = pipelet.process(getBlackboard(), recordIds);
    final int expectedSize = 5;
    assertEquals(expectedSize, matchingRecords.length);
    for (int i = 0; i < expectedSize; i++) {
      assertEquals(recordIds[i], matchingRecords[i]);
    }
  }

  /** tests error handling in case of record failures. */
  public void testRobustness() throws Exception {
    final FilterPipelet pipelet = new FilterPipelet(); 
    final AnyMap configuration = DataFactory.DEFAULT.createAnyMap();
    // create configuration where required parameter "filterAttribute" is missing
    configuration.put("filterExpression", ".+:[0-9]{1}"); // will match all records    
    pipelet.configure(configuration);

    final int noOfRecords = 9;
    int noOCorrectRecords = 0;
    final Collection<String> recordIds = new ArrayList<String>();
    for (int i = 1; i <= noOfRecords; i++) {
      final String recordId = createBlackboardRecord("filter", Integer.toString(i));
      recordIds.add(recordId);
      if (i % 2 == 0) { // prepare correct records        
        _blackboard.getRecord(recordId).getMetadata().getMap("_parameters", true)
          .put("filterAttribute", "_recordid");
        noOCorrectRecords++;
      }
    }
    doRobustnessTest(pipelet, configuration, recordIds, noOCorrectRecords, true);
    doRobustnessTest(pipelet, configuration, recordIds, noOCorrectRecords, false);
  }
}
