/*******************************************************************************
 * Copyright (c) 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets.test;

import java.io.InputStream;
import java.util.Arrays;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.StoreWriterPipelet;

/**
 * Tests the {@StoreWriterPipelet}.
 */
public class TestStoreWriterPipelet extends APipeletTest {

  private final String STORE_NAME = "teststore";

  private final DataFactory _dataFactoy = DataFactory.DEFAULT;

  private final IpcRecordReader _recordReader = new IpcRecordReader();

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    getObjectStoreService().ensureStore(STORE_NAME);
  }

  /**
   * test with one record where pipelet is configured via configuration.
   */
  public void testSingleInputRecord() throws Exception {
    final StoreWriterPipelet pipelet = new StoreWriterPipelet();
    final String objectId = "path1/path2";

    final AnyMap config = _dataFactoy.createAnyMap();
    config.put("store", STORE_NAME);
    config.put("objectId", objectId);
    pipelet.configure(config);

    final String recordId = createBlackboardRecord("testSource", "testKey");
    final Record inputRecord = _blackboard.getRecord(recordId);
    inputRecord.getMetadata().put("testAttribute", "testValue");

    final String[] ids = pipelet.process(getBlackboard(), new String[] { recordId });
    assertEquals(1, ids.length);
    assertTrue(Arrays.asList(ids).contains(recordId));

    final ObjectStoreService objectStore = getObjectStoreService();
    final InputStream instream = objectStore.readObject(STORE_NAME, objectId);
    byte[] recordAsBytes = IOUtils.toByteArray(instream);

    final Record recordFromStore = _recordReader.readBinaryObject(recordAsBytes);
    assertEquals(recordFromStore.getMetadata(), recordFromStore.getMetadata());
  }

  /**
   * test with multiple records where objectId parameter is configured via record parameters.
   */
  public void testMultipleInputRecords() throws Exception {
    final StoreWriterPipelet pipelet = new StoreWriterPipelet();

    final AnyMap config = _dataFactoy.createAnyMap();
    config.put("store", STORE_NAME);
    pipelet.configure(config);

    final String recordId1 = createBlackboardRecord("testSource", "1");
    final String recordId2 = createBlackboardRecord("testSource", "2");
    final String recordId3 = createBlackboardRecord("testSource", "3");
    final Record inputRecord1 = _blackboard.getRecord(recordId1);
    final Record inputRecord2 = _blackboard.getRecord(recordId2);
    final Record inputRecord3 = _blackboard.getRecord(recordId3);
    inputRecord1.getMetadata().put("testAttribute", "v1");
    inputRecord2.getMetadata().put("testAttribute", "v2");
    inputRecord3.getMetadata().put("testAttribute", "v3");

    inputRecord1.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true)
      .put("objectId", "objectId-1");
    inputRecord2.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true)
      .put("objectId", "objectId-2");
    inputRecord3.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true)
      .put("objectId", "objectId-3");

    final String[] ids = pipelet.process(getBlackboard(), new String[] { recordId1, recordId2, recordId3 });
    assertEquals(3, ids.length);
    assertTrue(Arrays.asList(ids).contains(recordId1));
    assertTrue(Arrays.asList(ids).contains(recordId2));
    assertTrue(Arrays.asList(ids).contains(recordId3));

    final ObjectStoreService objectStore = getObjectStoreService();

    byte[] recordAsBytes1 = IOUtils.toByteArray(objectStore.readObject(STORE_NAME, "objectId-1"));
    final Record recordFromStore1 = _recordReader.readBinaryObject(recordAsBytes1);
    assertEquals(recordFromStore1.getMetadata(), recordFromStore1.getMetadata());

    byte[] recordAsBytes2 = IOUtils.toByteArray(objectStore.readObject(STORE_NAME, "objectId-2"));
    final Record recordFromStore2 = _recordReader.readBinaryObject(recordAsBytes2);
    assertEquals(recordFromStore2.getMetadata(), recordFromStore2.getMetadata());

    byte[] recordAsBytes3 = IOUtils.toByteArray(objectStore.readObject(STORE_NAME, "objectId-3"));
    final Record recordFromStore3 = _recordReader.readBinaryObject(recordAsBytes3);
    assertEquals(recordFromStore3.getMetadata(), recordFromStore3.getMetadata());
  }

  private ObjectStoreService getObjectStoreService() throws Exception {
    final ObjectStoreService objectStore = getService(ObjectStoreService.class);
    assertNotNull("no ObjectStoreService service found.", objectStore);
    return objectStore;
  }
}
