/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation Drazen Cindric (Attensity Europe GmbH) -
 * data model improvements
 *******************************************************************************/

package org.eclipse.smila.processing.pipelets;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.MissingParameterException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * Simple pipelet that copies a String value between attributes and or attachments. It supports two modes: <li>COPY:
 * copies the value from the input to the output attribute/attachment</li> <li>MOVE: same as copy, but deletes the value
 * from the input attribute/attachment</li>
 */
public class CopyPipelet extends ATransformationPipelet {

  /**
   * property to configure the execution mode of the pipelet.
   */
  private static final String PROPERTY_MODE = "mode";

  /** The log. */
  private final Log _log = LogFactory.getLog(getClass());

  /** @return The execution mode. Default is copy. */
  protected Mode getMode(final ParameterAccessor paramAccessor) throws MissingParameterException {
    final String mode = paramAccessor.getParameter(PROPERTY_MODE, null);
    if (mode != null) {
      return Mode.valueOf(mode);
    }
    return Mode.COPY;
  }

  /** {@inheritDoc} */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    if (recordIds != null) {
      for (final String id : recordIds) {
        try {
          paramAccessor.setCurrentRecord(id);
          final String inputName = getInputName(paramAccessor);
          final SourceType inputType = getInputType(paramAccessor);

          copyInputToOutput(blackboard, id, paramAccessor, inputName, inputType);
          if (Mode.MOVE.equals(getMode(paramAccessor))) {
            removeInput(blackboard, id, inputName, inputType);
          }

          resultCollector.addResult(id);
        } catch (final Exception ex) {
          resultCollector.addFailedResult(id, ex);
        }
      }
    }
    return resultCollector.getResultIds();
  }

  /**
   * copy input value to output target. If copying from attribute to attribute, keep the type.
   * <p>
   * Convert attachments to String, or attribute values from String to byte[] as attachment.
   * 
   */
  private void copyInputToOutput(final Blackboard blackboard, final String id,
    final ParameterAccessor paramAccessor, final String inputName, final SourceType inputType)
    throws BlackboardAccessException, UnsupportedEncodingException, ProcessingException {
    final String outputName = getOutputName(paramAccessor);
    final SourceType outputType = getOutputType(paramAccessor);
    if (isReadFromAttribute(inputType) && isStoreInAttribute(outputType)) {
      final AnyMap metaData = blackboard.getMetadata(id);
      final Any attributeValue = metaData.get(inputName);
      if (attributeValue != null) {
        final ValueType outputValueType = getOutputValueType(paramAccessor);
        if (outputValueType != null) {
          final Value value = metaData.getFactory().parseFromString(attributeValue.asValue().asString(), outputValueType);
          metaData.put(outputName, value);
        } else {
          metaData.put(outputName, attributeValue);
        }
      }
    } else if (!isReadFromAttribute(inputType) && !isStoreInAttribute(outputType)) {
      final InputStream inputAttachment = blackboard.getAttachmentAsStream(id, inputName);
      if (inputAttachment != null) {
        try {
          blackboard.setAttachmentFromStream(id, outputName, inputAttachment);
        } finally {
          IOUtils.closeQuietly(inputAttachment);
        }
      }
    } else {
      final String srcValue = readStringInput(blackboard, id, paramAccessor);
      if (srcValue != null) {
        storeResult(blackboard, id, srcValue, paramAccessor);
        if (_log.isTraceEnabled()) {
          _log.trace("copied value from input " + inputType + " " + inputName + " to " + outputType + " "
            + outputName);
        }
      }
    }
  }

  private void removeInput(final Blackboard blackboard, final String id, final String inputName,
    final SourceType inputType) throws BlackboardAccessException {
    // remove old value
    if (isReadFromAttribute(inputType)) {
      blackboard.getMetadata(id).remove(inputName);
    } else {
      blackboard.removeAttachment(id, inputName);
    }
    if (_log.isTraceEnabled()) {
      _log.trace("deleted value of input " + inputType + " " + inputName);
    }
  }

  /**
   * Execution mode: copy or move value.
   */
  private enum Mode {
    /**
     * Copy the value.
     */
    COPY,

    /**
     * Move the value (deleting the value from the input source).
     */
    MOVE;
  }

}
