/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Tobias Liefke - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Attachment;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * This pipelet writes an attachment into a file described by a path attribute.
 * 
 * @author Tobias Liefke
 */
public class FileWriterPipelet implements Pipelet {

  /** The name of the property that contains the path attribute name. */
  public static final String PROPERTY_PATH_ATTRIBUTE = "pathAttribute";

  /** The name of the property that contains the content attachment name. */
  public static final String PROPERTY_CONTENT_ATTACHMENT = "contentAttachment";

  /** The name of the property that indicates to append the attachment to the file. */
  public static final String PROPERTY_APPEND = "append";

  /** The local log. */
  private final Log _log = LogFactory.getLog(getClass());

  /** the configuration. */
  private AnyMap _config;

  /**
   * {@inheritDoc}
   */
  @Override
  public void configure(final AnyMap configuration) throws ProcessingException {
    _config = configuration;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    for (final String id : recordIds) {
      try {
        paramAccessor.setCurrentRecord(id);
        final String contentAttachmentName = paramAccessor.getRequiredParameter(PROPERTY_CONTENT_ATTACHMENT);
        final Attachment attachment = blackboard.getAttachment(id, contentAttachmentName);
        if (attachment == null) {
          _log.warn("Attachment not found in record " + id + ": " + contentAttachmentName);
        } else {
          final String pathAttributeName = paramAccessor.getRequiredParameter(PROPERTY_PATH_ATTRIBUTE);
          final String path = blackboard.getMetadata(id).getStringValue(pathAttributeName);
          if (path == null) {
            _log.warn("File path not found in record " + id + ": " + pathAttributeName);
          } else {
            writeFile(attachment, path, paramAccessor.getBooleanParameter(PROPERTY_APPEND, Boolean.FALSE));
          }
        }
        resultCollector.addResult(id);
      } catch (final Exception e) {
        resultCollector.addFailedResult(id, new ProcessingException("Error in FileWriterPipelet processing ID "
          + id, e));
      }
    }
    return resultCollector.getResultIds();
  }

  /**
   * Writes the content of the given attachment to the file at the given path.
   * 
   * @param attachment
   *          the content attachment
   * @param path
   *          the path to the target file
   * @param append
   *          indicates to append the content to the file
   * @throws IOException
   *           if either a failure occurs during reading from the attachment or writing to the file
   */
  private void writeFile(final Attachment attachment, final String path, final boolean append) throws IOException {
    _log.debug((append ? "Writing to file " : "Creating file ") + path);
    final FileOutputStream out = new FileOutputStream(path, append);
    try {
      final InputStream in = attachment.getAsStream();
      try {
        IOUtils.copy(in, out);
      } finally {
        IOUtils.closeQuietly(in);
      }
    } finally {
      IOUtils.closeQuietly(out);
    }
  }
}
