/*******************************************************************************
 * Copyright (c) 2008, 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation, Andreas Schank (empolis GmbH)
 * additional tests
 **********************************************************************************************************************/

package org.eclipse.smila.processing.test;

import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Base class for WorkflowProcessor tests.
 * 
 * @author jschumacher
 * 
 */
public class TestPipeletTracker extends DeclarativeServiceTestCase {

  /**
   * PipeletTracker instance to test.
   */
  private PipeletTracker _tracker;

  /**
   * Check if WorkflowProcessor service is active. Wait up to 30 seconds for start. Fail, if no service is starting.
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _tracker = getService(PipeletTracker.class);
    assertNotNull("no PipeletTracker service found.", _tracker);
  }

  /**
   * test if pipelet tracker knows about Test1Pipelet and Test2Pipelet.
   * 
   * @throws Exception
   *           unexpected error.
   * 
   */
  public void testRegisteredPipelets() throws Exception {
    final Map<String, Class<? extends Pipelet>> pipelets = _tracker.getRegisteredPipelets();

    assertTrue(pipelets.containsKey(Test1Pipelet.class.getName()));
    final Class<? extends Pipelet> test1class = pipelets.get(Test1Pipelet.class.getName());
    final Pipelet test1pipelet = test1class.newInstance();
    test1pipelet.configure(null);
    assertTrue(test1pipelet instanceof Test1Pipelet);
    assertTrue(test1pipelet instanceof Pipelet);
    test1pipelet.process(null, null);

    assertTrue(pipelets.containsKey(Test2Pipelet.class.getName()));
    final Class<? extends Pipelet> test2class = pipelets.get(Test2Pipelet.class.getName());
    final Pipelet test2pipelet = test2class.newInstance();
    test2pipelet.configure(null);
    assertTrue(test2pipelet instanceof Test2Pipelet);
    assertTrue(test2pipelet instanceof Pipelet);
    test2pipelet.process(null, null);

    assertTrue(pipelets.containsKey(Test3Pipelet.class.getName()));
    final Class<? extends Pipelet> test3class = pipelets.get(Test3Pipelet.class.getName());
    final Pipelet test3pipelet = test3class.newInstance();
    test3pipelet.configure(null);
    assertTrue(test3pipelet instanceof Test3Pipelet);
    assertTrue(test3pipelet instanceof Pipelet);
    test3pipelet.process(null, null);

    // invalid pipelet description, the pipelet won't have been loaded...
    assertFalse(pipelets.containsKey(Test4Pipelet.class.getName()));
  }

  /**
   * Tests if the PipeletTracker returns the descriptions.
   * 
   * @throws Exception
   *           something went terribly wrong.
   */
  public void testRegisteredPipeletDescriptions() throws Exception {
    final String descriptionPrefix = "test pipelet number ";
    final String descriptionPostfix = ".";
    final String descriptionKey = "description";
    final String parametersKey = "parameters";
    final String errorsKey = "errors";
    AnySeq parameters;
    final Map<String, AnyMap> pipeletDescriptions = _tracker.getRegisteredPipeletDescriptions();

    assertTrue(pipeletDescriptions.containsKey(Test1Pipelet.class.getName()));
    String testdescription = pipeletDescriptions.get(Test1Pipelet.class.getName()).getStringValue(descriptionKey);
    assertEquals(descriptionPrefix + "1" + descriptionPostfix, testdescription);
    parameters = pipeletDescriptions.get(Test1Pipelet.class.getName()).getSeq(parametersKey);
    assertNotNull(parameters);
    assertFalse(pipeletDescriptions.get(Test1Pipelet.class.getName()).containsKey(errorsKey));
    assertFalse(parameters.isEmpty());
    AnyMap param1 = parameters.getMap(0);
    if (!"param1".equals(param1.getStringValue("name"))) {
      param1 = parameters.getMap(1);
    }
    assertEquals(param1.getStringValue("name"), "param1");
    assertEquals(param1.getStringValue("type"), "string");
    assertEquals(param1.getBooleanValue("optional"), Boolean.TRUE);
    assertTrue(param1.getSeq("values").contains(DataFactory.DEFAULT.createStringValue("val1")));
    assertEquals("val2", param1.getSeq("values").getMap(1).getStringValue("value"));
    AnyMap param2 = parameters.getMap(1);
    if (!"param2".equals(param2.getStringValue("name"))) {
      param2 = parameters.getMap(0);
    }
    assertEquals(param2.getStringValue("name"), "param2");
    assertEquals(param2.getStringValue("type"), "double");

    assertTrue(pipeletDescriptions.containsKey(Test2Pipelet.class.getName()));
    testdescription = pipeletDescriptions.get(Test2Pipelet.class.getName()).getStringValue(descriptionKey);
    assertEquals(descriptionPrefix + "2" + descriptionPostfix, testdescription);
    parameters = pipeletDescriptions.get(Test2Pipelet.class.getName()).getSeq(parametersKey);
    assertNull(parameters);
    assertFalse(pipeletDescriptions.get(Test2Pipelet.class.getName()).containsKey(errorsKey));

    assertTrue(pipeletDescriptions.containsKey(Test3Pipelet.class.getName()));
    testdescription = pipeletDescriptions.get(Test3Pipelet.class.getName()).getStringValue(descriptionKey);
    assertEquals(descriptionPrefix + "3" + descriptionPostfix, testdescription);
    parameters = pipeletDescriptions.get(Test3Pipelet.class.getName()).getSeq(parametersKey);
    assertNotNull(parameters);
    assertTrue(parameters.isEmpty());
    assertFalse(pipeletDescriptions.get(Test3Pipelet.class.getName()).containsKey(errorsKey));

    // number 4 has invalid json description file, it must not be listed:
    assertFalse(pipeletDescriptions.containsKey(Test4Pipelet.class.getName()));
  }

  /**
   * Tests that an errors entry will be generated for pipelets that cannot be found.
   * 
   * @throws Exception
   *           test failed.
   */
  public void testErrorIfPipeletCannotbeFound() throws Exception {
    final String errorsKey = "errors";
    final String descriptionKey = "description";
    final String className = "org.eclipse.smila.processing.test.WillNotBeFoundPipelet";
    final Map<String, AnyMap> pipeletDescriptions = _tracker.getRegisteredPipeletDescriptions();

    assertTrue(pipeletDescriptions.containsKey(className));
    final String testdescription = pipeletDescriptions.get(className).getStringValue(descriptionKey);
    assertNotNull(testdescription);
    final AnySeq errors = pipeletDescriptions.get(className).getSeq(errorsKey);
    assertEquals(1, errors.size());
  }

  /**
   * Tests that an errors entry will be generated for pipelets that have invalid parameters sections.
   * 
   * @throws Exception
   *           test failed.
   */
  public void testErrorIfParametersAreInvalid() throws Exception {
    final String errorsKey = "errors";
    final String descriptionKey = "description";
    final String className1 = "org.eclipse.smila.processing.test.InvalidParam1Pipelet";
    final String className2 = "org.eclipse.smila.processing.test.InvalidParam2Pipelet";
    final Map<String, AnyMap> pipeletDescriptions = _tracker.getRegisteredPipeletDescriptions();

    assertTrue(pipeletDescriptions.containsKey(className1));
    // but it is registered...
    assertTrue(_tracker.getRegisteredPipelets().containsKey(className1));
    String testdescription = pipeletDescriptions.get(className1).getStringValue(descriptionKey);
    assertNotNull(testdescription);
    AnySeq errors = pipeletDescriptions.get(className1).getSeq(errorsKey);
    assertEquals(1, errors.size());

    assertTrue(pipeletDescriptions.containsKey(className2));
    // but it is registered...
    assertTrue(_tracker.getRegisteredPipelets().containsKey(className2));
    testdescription = pipeletDescriptions.get(className2).getStringValue(descriptionKey);
    assertNotNull(testdescription);
    errors = pipeletDescriptions.get(className2).getSeq(errorsKey);
    assertEquals(1, errors.size());
  }
}
