/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank, Andreas Weber(Attensity Europe GmbH) - implementation
 *******************************************************************************/
package org.eclipse.smila.processing.worker.test;

import java.io.IOException;
import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.processing.pipelets.AddValuesPipelet;
import org.eclipse.smila.processing.worker.ProcessingWorker;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.input.RecordInput;

public class TestPipeletProcessorWorker extends ProcessingWorkerTestBase {

  /** Test with a pipelet name that doesn't exist. */
  public void testErrorOnNonExistingPipelet() throws Exception {
    final String workflowName = "testPipeletProcessor";
    final int noOfRecords = 10;
    final String jobName = "testJobError";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("pipeletName", PipeletProcessorWorkerTestPipelet.class.getName() + "DoesNotExist");
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    addJob(jobName, workflowName, parameters);
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, DataFactory.DEFAULT.createRecord(Integer.toString(i)));
    }
    _builder.commitJob(jobName);
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId, 100000); // wait max. 10 seconds
    assertJobRunFailed(jobName, jobRunId);
  }

  /** Tests if records have been processed successfully. */
  public void testRecordsProcessed() throws Exception {
    final String workflowName = "testPipeletProcessor";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords);
    checkAndRemoveRecords(STORE, noOfRecords);
  }

  /** Test with complex parameters. */
  public void testComplexParameters() throws Exception {
    final String workflowName = "testPipeletProcessor";
    final int noOfRecords = 10;
    final String jobName = "testJobError";
    final String outAttributeName = "outAttribute";
    final String[] outAttributeValues = { "pipeletName=${pipeletName}.", "outValue2", "outValue3" };
    final String[] expectedOutAttributeValues =
      { "pipeletName=" + AddValuesPipelet.class.getName() + ".", "outValue2", "outValue3" };
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("pipeletName", AddValuesPipelet.class.getName());
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("outputAttribute", outAttributeName);
    for (final String outAttributeValue : outAttributeValues) {
      parameters.add("valuesToAdd", DataFactory.DEFAULT.createStringValue(outAttributeValue));
    }
    parameters.put("store", STORE);
    addJob(jobName, workflowName, parameters);
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, DataFactory.DEFAULT.createRecord(Integer.toString(i)));
    }
    commitAndWaitForJob(jobName, jobRunId, true);
    // now check for the records that have been produced...
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE, "outbulkBucket");
    int recordsFound = 0;
    for (final StoreObject storeObject : storeObjects) {
      final RecordInput rinput =
        new RecordInput(new BulkInfo("outbulkBucket", STORE, storeObject.getId()), _objectStoreService);
      try {
        Record record = null;
        do {
          record = rinput.getRecord();
          if (record != null) {
            ++recordsFound;
            final AnySeq addedOutputAttribute = record.getMetadata().getSeq(outAttributeName);
            assertNotNull(addedOutputAttribute);
            assertEquals(expectedOutAttributeValues.length, addedOutputAttribute.size());
            for (final String val : expectedOutAttributeValues) {
              assertTrue(addedOutputAttribute.contains(DataFactory.DEFAULT.createStringValue(val)));
            }
          }
        } while (record != null);
      } finally {
        rinput.close();
      }
    }
    assertEquals(noOfRecords, recordsFound);
  }

  @Override
  protected void createJob(final String workflowName, final String jobName) throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    parameters.put("pipeletName", PipeletProcessorWorkerTestPipelet.class.getName());
    addJob(jobName, workflowName, parameters);
  }

  /**
   * checks if the respective attribute has been filled by the PipeletProcessorWorkerTestPipelet.
   * 
   * @throws ObjectStoreException
   * @throws IOException
   */
  private void checkAndRemoveRecords(final String storeName, final int noOfRecords) throws ObjectStoreException,
    IOException {
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(storeName, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, storeName, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    int recordCount = 0;
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        recordCount++;
        assertEquals(PipeletProcessorWorkerTestPipelet.TEST_TEXT,
          record.getMetadata().getStringValue(PipeletProcessorWorkerTestPipelet.ATTRIBUTE_NAME));
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
      _objectStoreService.removeObject(storeName, bulkInfo.getObjectName());
    }
    assertEquals(noOfRecords, recordCount);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingDefaultSettings() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    final int noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, noOfRecordsProcessedParallel, null, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingWriteAttachments() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    final int noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, noOfRecordsProcessedParallel, null, true);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingDropAttachments() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    final int noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, noOfRecordsProcessedParallel, null, false);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, false, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingAttachmentsInMemory() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    final int noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, noOfRecordsProcessedParallel, true, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingAttachmentsInBinStorage() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    final int noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, noOfRecordsProcessedParallel, false, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, false);
  }

  private void createJobForAttachments(final String jobName, final int noOfRecordsProcessedParallel,
    final Boolean attachmentsInMemory, final Boolean writeAttachments) throws Exception {
    final String workflowName = "testPipeletProcessor";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    parameters.put("pipeletName", BpelWorkerAttachmentTestPipelet.class.getName());
    if (attachmentsInMemory != null) {
      parameters.put(ProcessingWorker.KEY_KEEPATTACHMENTSINMEMORY, attachmentsInMemory);
    }
    if (writeAttachments != null) {
      parameters.put(ProcessingWorker.KEY_WRITEATTACHMENTSTOOUTPUT, writeAttachments);
    }
    addJob(jobName, workflowName, parameters);
  }

}
