/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank, Andreas Weber(Attensity Europe GmbH) - implementation
 *******************************************************************************/
package org.eclipse.smila.processing.worker.test;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.worker.PipelineProcessorWorker;
import org.eclipse.smila.processing.worker.ProcessingWorker;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.input.RecordInput;

public class TestPipelineProcessorWorker extends ProcessingWorkerTestBase {

  private static final String BPEL_TEST_PIPELINE = "TestPipeline";

  private int _noOfRecordsProcessedParallel;

  /** Test with a pipeline name that doesn't exist. */
  public void testErrorOnNonExistingPipeline() throws Exception {
    final String workflowName = "testWithoutOutputBucket";
    final int noOfRecords = 10;
    final String jobName = "testJobError";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(PipelineProcessorWorker.KEY_PIPELINE_NAME, "PipelineThatDoesNotExist");
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    addJob(jobName, workflowName, parameters);
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, DataFactory.DEFAULT.createRecord(Integer.toString(i)));
    }
    _builder.commitJob(jobName);
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId, 100000); // wait max. 10 seconds
    assertJobRunFailed(jobName, jobRunId);
  }

  /** Test with a workflow containing a BPEL worker that has no output bucket. */
  public void testWithoutOutputBucket() throws Exception {
    final String workflowName = "testWithoutOutputBucket";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords, 1);
    executeJob(workflowName, noOfRecords, 5);
    executeJob(workflowName, noOfRecords, 10);
    executeJob(workflowName, noOfRecords, 20);
    executeJob(workflowName, noOfRecords, 0);
    executeJob(workflowName, noOfRecords, -1);
  }

  /** Test with a workflow containing a BPEL worker that has an output bucket. */
  public void testWithOutputBucket() throws Exception {
    final String workflowName = "testWithOutputBucket";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords, 1);
    checkAndRemoveRecords(STORE, noOfRecords, 1);
    executeJob(workflowName, noOfRecords, 5);
    checkAndRemoveRecords(STORE, noOfRecords, 5);
    executeJob(workflowName, noOfRecords, 10);
    checkAndRemoveRecords(STORE, noOfRecords, 10);
    executeJob(workflowName, noOfRecords, 20);
    checkAndRemoveRecords(STORE, noOfRecords, 20);
    executeJob(workflowName, noOfRecords, 0);
    checkAndRemoveRecords(STORE, noOfRecords, PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE);
    executeJob(workflowName, noOfRecords, -1);
    checkAndRemoveRecords(STORE, noOfRecords, PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE);
  }

  public void testTaskParametersInRecord() throws Exception {
    final String workflowName = "testWithOutputBucket";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords, 1);
    checkParametersAndRemoveRecords(STORE, noOfRecords, 1);
    executeJob(workflowName, noOfRecords, 5);
    checkParametersAndRemoveRecords(STORE, noOfRecords, 5);
    executeJob(workflowName, noOfRecords, 10);
    checkParametersAndRemoveRecords(STORE, noOfRecords, 10);
  }

  /** Tests if _failOnError is set correctly */
  public void testFailOnErrorParameterInRecord() throws Exception {
    String workflowName = "testWithOutputBucket";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords, 1);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 1, "false");
    executeJob(workflowName, noOfRecords, 5);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 5, "false");
    executeJob(workflowName, noOfRecords, 10);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 10, "false");
    workflowName = "testWithOutputBucketAndFailOnError";
    executeJob(workflowName, noOfRecords, 1);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 1, "true");
    executeJob(workflowName, noOfRecords, 5);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 5, "true");
    executeJob(workflowName, noOfRecords, 10);
    checkFailOnErrorAndRemoveRecords(STORE, noOfRecords, 10, "true");
  }

  /** Test with a workflow containing two BPEL worker that has an output bucket. */
  public void testTwoPipelinesWithOutputBucket() throws Exception {
    final String workflowName = "testMoreThanOneWorkerWithOutputBucket";
    final int noOfRecords = 10;
    executeJob(workflowName, noOfRecords, 1);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, 1);
    executeJob(workflowName, noOfRecords, 5);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, 5);
    executeJob(workflowName, noOfRecords, 10);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, 10);
    executeJob(workflowName, noOfRecords, 20);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, 20);
    executeJob(workflowName, noOfRecords, 0);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE);
    executeJob(workflowName, noOfRecords, -1);
    checkAndRemoveRecordsForTwoPipelines(STORE, noOfRecords, PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE);
  }

  /** tests handling of recoverable ProcessingExceptions */
  public void testProcessingRecoverableException() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testJobExceptionsRecoverable";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    final int noOfErrorRecord = 5;
    final String workflowName = "testWithOutputBucket";
    createJob(workflowName, jobName);
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      final Record record = DataFactory.DEFAULT.createRecord(Integer.toString(i));
      if (i == noOfErrorRecord) {
        record.getMetadata().put(BpelWorkerTestPipelet.ATTRIBUTE_THROW_EXCEPTION, true);
        record.getMetadata().put(BpelWorkerTestPipelet.ATTRIBUTE_THROW_RECOVERABLE_EXCEPTION, true);
      }
      _builder.addRecord(jobName, record);
    }
    commitAndWaitForJob(jobName, jobRunId, true);
    // everything should be ok, no record will be missing...
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, STORE, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    final Collection<String> ids = new HashSet<String>();
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        ids.add(record.getId());
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
    }
    assertEquals(noOfRecords, ids.size());
  }

  /** tests handling of non-recoverable ProcessingExceptions of one record. */
  public void testProcessingNonrecoverableExceptionOfOneRecord() throws Exception {
    final String jobName = "testProcessingNonrecoverableException";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    final int noOfErrorRecord = 5;
    final String workflowName = "testWithOutputBucket";
    createJob(workflowName, jobName);
    // nonrecoverable ProcessingException
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      final Record record = DataFactory.DEFAULT.createRecord(Integer.toString(i));
      if (i == noOfErrorRecord) {
        record.getMetadata().put(BpelWorkerTestPipelet.ATTRIBUTE_THROW_EXCEPTION, true);
      }
      _builder.addRecord(jobName, record);
    }
    commitAndWaitForJob(jobName, jobRunId, true);
    // everything should be ok, but one bunch of records will be missing...
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, STORE, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    final Collection<String> ids = new HashSet<String>();
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        ids.add(record.getId());
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
    }
    // one bunch of parallely processed records will be missing...
    assertEquals(noOfRecords - _noOfRecordsProcessedParallel, ids.size());
  }

  /** tests handling of non-recoverable ProcessingExceptions of all records. */
  public void testProcessingNonrecoverableExceptionOfAllRecords() throws Exception {
    final String jobName = "testProcessingNonrecoverableException";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    final String workflowName = "testWithOutputBucket";
    createJob(workflowName, jobName);
    // nonrecoverable ProcessingException
    final String jobRunId = startJob(jobName);
    for (int i = 0; i < noOfRecords; i++) {
      final Record record = DataFactory.DEFAULT.createRecord(Integer.toString(i));
      record.getMetadata().put(BpelWorkerTestPipelet.ATTRIBUTE_THROW_EXCEPTION, true);
      _builder.addRecord(jobName, record);
    }
    commitAndWaitForJob(jobName, jobRunId, false); // false -> job run should be failed, no successful workflow run
    // there should be no ouput objects
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(STORE, BUCKET_NAME);
    assertEquals(0, storeObjects.size());
  }

  private void executeJob(final String workflowName, final int noOfRecords, final int noOfRecordsProcessedParallel)
    throws Exception {
    _noOfRecordsProcessedParallel = noOfRecordsProcessedParallel;
    executeJob(workflowName, noOfRecords);
    if (noOfRecordsProcessedParallel <= 0) {
      assertEquals("Expecting default " + PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE + " for value <= 0",
        PipelineProcessorWorker.DEFAULT_PIPELINERUN_BULKSIZE, BpelWorkerTestPipelet._lastParallelRecordCount);
    } else if (noOfRecords < noOfRecordsProcessedParallel) {
      assertEquals(noOfRecords, BpelWorkerTestPipelet._lastParallelRecordCount);
    } else {
      assertEquals(_noOfRecordsProcessedParallel, BpelWorkerTestPipelet._lastParallelRecordCount);
    }
  }

  @Override
  protected void createJob(final String workflowName, final String jobName) throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(PipelineProcessorWorker.KEY_PIPELINERUN_BULKSIZE, _noOfRecordsProcessedParallel);
    parameters.put(PipelineProcessorWorker.KEY_PIPELINE_NAME, BPEL_TEST_PIPELINE);
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    addJob(jobName, workflowName, parameters);
  }

  private void checkAndRemoveRecordsForTwoPipelines(final String store2, final int noOfRecords,
    final int noOfParallelRecords) throws ObjectStoreException, IOException {
    final Collection<Long> parallelRecordNos = new ArrayList<Long>();
    parallelRecordNos.add(Long.valueOf(Math.min(noOfParallelRecords, noOfRecords)));
    // 1: constant from workflow.
    parallelRecordNos.add(Long.valueOf(Math.min(1, noOfRecords)));
    checkAndRemoveRecords(STORE, noOfRecords, parallelRecordNos, Math.min(noOfParallelRecords, noOfRecords));
  }

  private void checkAndRemoveRecords(final String store2, final int noOfRecords, final int noOfParallelRecords)
    throws ObjectStoreException, IOException {
    checkAndRemoveRecords(STORE, noOfRecords, null, Math.min(noOfParallelRecords, noOfRecords));
  }

  /**
   * checks the output bulk for correctness and removes it afterwards.
   * 
   * @throws ObjectStoreException
   * @throws IOException
   */
  private void checkAndRemoveRecords(final String storeName, final int noOfRecords,
    final Collection<Long> noOfParallelRecords, final int attributeRecordNo) throws ObjectStoreException,
    IOException {
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(storeName, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, storeName, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    final Collection<String> ids = new HashSet<String>();
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        ids.add(record.getId());
        if (noOfParallelRecords != null) {
          final Collection<Long> noOfParallelRecordsFound =
            record.getMetadata().getSeq(BpelWorkerTestPipelet2.SECOND_ATTRIBUTE).asLongs();
          assertEquals(noOfParallelRecords, noOfParallelRecordsFound);
        }
        assertEquals(attributeRecordNo, record.getMetadata().getSeq(BpelWorkerTestPipelet2.ATTRIBUTE_RECORD_COUNT)
          .getLongValue(0).intValue());
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
      _objectStoreService.removeObject(storeName, bulkInfo.getObjectName());
    }
  }

  /**
   * checks the output bulk for correctness and removes it afterwards.
   * 
   * @throws ObjectStoreException
   * @throws IOException
   */
  private void checkParametersAndRemoveRecords(final String storeName, final int noOfRecords,
    final int noOfParallelRecords) throws ObjectStoreException, IOException {
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(storeName, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, storeName, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    int recordCount = 0;
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        recordCount++;
        final AnyMap parameters = record.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE);
        assertNotNull(parameters);
        // compare with #createJob() for expected parameter values.
        assertEquals(BPEL_TEST_PIPELINE, parameters.getStringValue(PipelineProcessorWorker.KEY_PIPELINE_NAME));
        assertEquals(Integer.toString(noOfParallelRecords),
          parameters.getStringValue(PipelineProcessorWorker.KEY_PIPELINERUN_BULKSIZE));
        assertEquals(TEMP_STORE, parameters.getStringValue("tempStore"));
        assertEquals(STORE, parameters.getStringValue("store"));
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
      _objectStoreService.removeObject(storeName, bulkInfo.getObjectName());
    }
    assertEquals(noOfRecords, recordCount);
  }

  /**
   * checks the _failOnError parameter and removes the records afterwards.
   * 
   * @throws ObjectStoreException
   * @throws IOException
   */
  private void checkFailOnErrorAndRemoveRecords(final String storeName, final int noOfRecords,
    final int noOfParallelRecords, final String failOnError) throws ObjectStoreException, IOException {
    final Collection<StoreObject> storeObjects = _objectStoreService.getStoreObjectInfos(storeName, BUCKET_NAME);
    assertEquals(1, storeObjects.size());
    final BulkInfo bulkInfo = new BulkInfo(BUCKET_NAME, storeName, storeObjects.iterator().next().getId());
    final RecordInput recordInput = new RecordInput(bulkInfo, _objectStoreService);
    int recordCount = 0;
    try {
      Record record = recordInput.getRecord();
      while (record != null) {
        recordCount++;
        final AnyMap parameters = record.getMetadata().getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE);
        assertNotNull(parameters);
        assertEquals(failOnError, parameters.getStringValue(ProcessingConstants.KEY_FAIL_ON_ERROR));
        record = recordInput.getRecord();
      }
    } finally {
      recordInput.close();
      _objectStoreService.removeObject(storeName, bulkInfo.getObjectName());
    }
    assertEquals(noOfRecords, recordCount);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingDefaultSettings() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, null, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingWriteAttachments() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, null, true);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingDropAttachments() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, null, false);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, false, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingAttachmentsInMemory() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, true, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, true);
  }

  /** tests handling of attachments, attachments should be kept. */
  public void testAttachmentProcessingAttachmentsInBinStorage() throws Exception {
    // Recoverable ProcessingException
    final String jobName = "testProcessingWithAttachments";
    final String attachmentName = "attachment";
    _noOfRecordsProcessedParallel = 5;
    final int noOfRecords = 10;
    createJobForAttachments(jobName, false, null);
    final String jobRunId = startJob(jobName);
    pushRecordsWithAttachment(jobName, jobRunId, noOfRecords, attachmentName);
    // everything should be ok, no record will be missing...
    assertRecordsWithAttachmentsProcessed(noOfRecords, attachmentName, true, false);
  }

  private void createJobForAttachments(final String jobName, final Boolean attachmentsInMemory,
    final Boolean writeAttachments) throws Exception {
    final String workflowName = "testWithOutputBucket";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(PipelineProcessorWorker.KEY_PIPELINERUN_BULKSIZE, _noOfRecordsProcessedParallel);
    parameters.put(PipelineProcessorWorker.KEY_PIPELINE_NAME, "TestAttachmentsPipeline");
    parameters.put("tempStore", TEMP_STORE);
    parameters.put("store", STORE);
    if (attachmentsInMemory != null) {
      parameters.put(ProcessingWorker.KEY_KEEPATTACHMENTSINMEMORY, attachmentsInMemory);
    }
    if (writeAttachments != null) {
      parameters.put(ProcessingWorker.KEY_WRITEATTACHMENTSTOOUTPUT, writeAttachments);
    }
    addJob(jobName, workflowName, parameters);
  }

}
