/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.solr.SolrServerService;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

public class TestApplicationScripts {

  private ScriptingEngine _engine;

  private SolrServer _solrServer;

  @Before
  public void setup() throws Exception {
    _engine = ServiceUtils.getService(ScriptingEngine.class);
    _solrServer = ServiceUtils.getService(SolrServerService.class).getServer("collection1");
    _solrServer.deleteByQuery("*:*");
    commitToIndex();
  }

  private void commitToIndex() throws SolrServerException, IOException {
    _solrServer.commit(true, true);
  }

  @Test
  public void testSmilaScriptCatalog() throws Exception {
    final List<String> expectedNames =
      new ArrayList<String>(Arrays.asList(new String[] { "add.process", "delete.process", "search.process" }));
    final AnySeq catalog = _engine.listScripts();
    assertNotNull(catalog);
    assertFalse(catalog.isEmpty());
    for (final Any entry : catalog) {
      assertTrue(entry.asMap().containsKey("name"));
      assertTrue(entry.asMap().containsKey("description"));
      if (expectedNames.remove(entry.asMap().getStringValue("name"))) {
        assertEquals("_SMILA_UNIT_TESTS_", entry.asMap().getStringValue("version"));
      }
    }
    assertTrue("Missing catalog entries for: " + expectedNames, expectedNames.isEmpty());
  }

  @Test
  public void testIndexBuildAndSearch() throws Exception {
    final Record addRecord = DataFactory.DEFAULT.createRecord("testIndexBuildAndSearch");
    addRecord.getMetadata().put("Filename", "testIndexBuildAndSearch.html");
    addRecord.getMetadata().put("MimeType", "text/html");
    addRecord.getMetadata().put("Extension", "html");
    addRecord.getMetadata().put("Size", "42");
    addRecord.getMetadata().put("LastModifiedDate", DataFactory.DEFAULT.createDateTimeValue(new Date()));
    addRecord.setAttachment("Content",
      "<html><head><title>Hello World</title></head> <body> Indexed by script</body></html>".getBytes());
    final Record addResult = _engine.callScript("application/add.process", addRecord);
    assertNotNull(addResult);
    commitToIndex();

    final AnyMap query = DataFactory.DEFAULT.createAnyMap();
    query.put("query", "script");
    query.put("QueryAttribute", "Content");
    query.getSeq("resultAttributes", true).add("_recordid");
    query.getSeq("resultAttributes", true).add("Content");
    query.getSeq("resultAttributes", true).add("Title");
    query.getSeq("resultAttributes", true).add("Filename");
    query.getSeq("resultAttributes", true).add("MimeType");
    query.getSeq("resultAttributes", true).add("Extension");
    query.getSeq("resultAttributes", true).add("Size");
    query.getSeq("resultAttributes", true).add("LastModifiedDate");

    final AnyMap result = _engine.callScript("application/search.process", query);
    assertNotNull(result);
    assertEquals(1, result.getLongValue("count").intValue());
    final AnyMap foundRecord = result.getSeq("records").getMap(0);

    assertEquals(addRecord.getMetadata().get("Filename"), foundRecord.get("Filename"));
    assertEquals(addRecord.getMetadata().get("MimeType"), foundRecord.get("MimeType"));
    assertEquals(addRecord.getMetadata().get("Extension"), foundRecord.get("Extension"));
    assertEquals(addRecord.getMetadata().getLongValue("Size"), foundRecord.getLongValue("Size"));
    assertEquals(addRecord.getMetadata().get("LastModifiedDate"), foundRecord.get("LastModifiedDate"));
    assertEquals("Hello World", foundRecord.getSeq("Title").getStringValue(0));
    assertTrue(foundRecord.getMap("_highlight").getMap("Content").getStringValue("text").contains("<b>script</b>"));

    final Record delResult = _engine.callScript("application/delete.process", addRecord);
    assertNotNull(delResult);
    commitToIndex();

    // SolrIndexPipelet does not support deleting records ... when it does we can use this code to check.
    final AnyMap result2 = _engine.callScript("application/search.process", query);
    assertNotNull(result2);
    assertEquals(0, result.getLongValue("count").intValue());
  }

  @Test
  public void testAddWithXmlSplit() throws Exception {
    final File xmlFile = ConfigUtils.getConfigFile("data", "split.xml");
    final Record addRecord = DataFactory.DEFAULT.createRecord("testAddWithXmlSplit");
    addRecord.getMetadata().put("Path", xmlFile.getAbsolutePath().toString());
    addRecord.getMetadata().put("Extension", "xml");
    addRecord.getMetadata().put("Size", "42");
    addRecord.getMetadata().put("LastModifiedDate", DataFactory.DEFAULT.createDateTimeValue(new Date()));
    final Record addResult = _engine.callScript("application/addWithXmlSplit.processRecord", addRecord);
    assertNotNull(addResult);
    commitToIndex();

    final AnyMap query = DataFactory.DEFAULT.createAnyMap();
    query.put("query", "*:*");
    query.getSeq("resultAttributes", true).add("_recordid");
    query.getSeq("resultAttributes", true).add("Content");
    query.getSeq("resultAttributes", true).add("Title");

    final AnyMap result = _engine.callScript("application/search.process", query);
    assertNotNull(result);
    assertEquals(3, result.getLongValue("count").intValue());
    for (final Any record : result.getSeq("records")) {
      assertTrue(record.asMap().getSeq("Title").getStringValue(0).startsWith("title"));
      assertTrue(record.asMap().getStringValue("Content").startsWith("text"));
    }
  }

  // @Test
  public void testManyManyRequests() throws Exception {
    final Record addRecord = DataFactory.DEFAULT.createRecord("testIndexBuildAndSearch");
    addRecord.getMetadata().put("Filename", "testIndexBuildAndSearch.html");
    addRecord.getMetadata().put("MimeType", "text/html");
    addRecord.getMetadata().put("Extension", "html");
    addRecord.getMetadata().put("Size", "42");
    addRecord.getMetadata().put("LastModifiedDate", DataFactory.DEFAULT.createDateTimeValue(new Date()));
    addRecord.setAttachment("Content",
      "<html><head><title>Hello World</title></head> <body> Indexed by script</body></html>".getBytes());
    final Record addResult = _engine.callScript("application/add.process", addRecord);
    assertNotNull(addResult);
    commitToIndex();

    final AnyMap query = DataFactory.DEFAULT.createAnyMap();
    query.put("query", "script");
    query.put("QueryAttribute", "Content");
    query.getSeq("resultAttributes", true).add("Content");
    query.getSeq("resultAttributes", true).add("Title");
    query.getSeq("resultAttributes", true).add("Filename");
    query.getSeq("resultAttributes", true).add("MimeType");
    query.getSeq("resultAttributes", true).add("Extension");
    query.getSeq("resultAttributes", true).add("Size");
    query.getSeq("resultAttributes", true).add("LastModifiedDate");

    for (int i = 0; i < Integer.MAX_VALUE; i++) {
      final AnyMap result = _engine.callScript("application/search.process", query);
      assertNotNull(result);
      assertEquals(1, result.getLongValue("count").intValue());
      if (i % 1000 == 0) {
        final Runtime r = Runtime.getRuntime();
        System.out.printf("After %d requests: Memory free %.1f, total %.1f, max %.1f\n", i, r.freeMemory() / 1e6,
          r.totalMemory() / 1e6, r.maxMemory() / 1e6);
      }
    }

  }
}
