/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.scripting.internal.JavascriptEngine;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;
import org.mozilla.javascript.EvaluatorException;

public class TestAttachments {
  private ScriptingEngine _engine;

  @Before
  public void setup() throws Exception {
    _engine = ServiceUtils.getService(ScriptingEngine.class);
  }

  @Test
  public void testReadAttachment() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.setAttachment("attachment", "attachment content".getBytes("utf-8"));
    final Record out = _engine.callScript("testAttachments.getAttachment", in);
    assertEquals("attachment content".length(), out.getMetadata().getLongValue("size").intValue());
    assertEquals("object", out.getMetadata().getStringValue("typeOfBytes"));
    assertEquals("object", out.getMetadata().getStringValue("typeOfStream"));
  }

  @Test
  public void testCopyAttachmentToSameName() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.setAttachment("the-attachment", "attachment content".getBytes("utf-8"));
    in.getMetadata().put("name", "the-attachment");
    in.getMetadata().put("newName", "the-attachment");
    final Record out = _engine.callScript("testAttachments.copyAttachment", in);
    assertEquals("attachment content", new String(out.getAttachmentAsBytes("the-attachment"), "utf-8"));
    assertEquals("attachment content".length(), out.getMetadata().getLongValue("size").intValue());
  }

  @Test
  public void testCopyAttachmentToNewName() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.setAttachment("original", "attachment content".getBytes("utf-8"));
    in.getMetadata().put("name", "original");
    in.getMetadata().put("newName", "copy");
    final Record out = _engine.callScript("testAttachments.copyAttachment", in);
    assertEquals("attachment content", new String(out.getAttachmentAsBytes("original"), "utf-8"));
    assertEquals("attachment content", new String(out.getAttachmentAsBytes("copy"), "utf-8"));
    assertEquals("attachment content".length(), out.getMetadata().getLongValue("size").intValue());
  }

  @Test
  public void testSetAttachmentFromString() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("name", "text");
    in.getMetadata().put("content", "ättächment cöntent");
    final Record out = _engine.callScript("testAttachments.setAttachmentFromString", in);
    assertEquals("Attachment: ättächment cöntent", new String(out.getAttachmentAsBytes("text"), "utf-8"));
  }

  @Test
  public void testSetAttachmentFromStream() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("name", "filecontent");
    in.getMetadata().put("filename",
      ConfigUtils.getConfigFile(JavascriptEngine.BUNDLE_NAME, "js/testAttachments.js").getAbsolutePath());
    final Record out = _engine.callScript("testAttachments.setAttachmentFromStream", in);
    final String expectedContent =
      ConfigUtils.getConfigContent(JavascriptEngine.BUNDLE_NAME, "js/testAttachments.js");
    assertEquals(expectedContent, new String(out.getAttachmentAsBytes("filecontent"), "utf-8"));
  }

  @Test
  public void testSetAttachmentFromInvalidType() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("hello", "world");
    try {
      _engine.callScript("testAttachments.setAttachmentFromInvalidType", in);
      fail("should not work");
    } catch (final ScriptingEngineException ex) {
      assertTrue("Got " + ex.getCause(), ex.getCause() instanceof EvaluatorException);
      System.out.println(ex.getCause());
    }
  }

  @Test
  public void testDeleteAttachment() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("name", "Content");
    in.setAttachment("Content", "attachment content".getBytes("utf-8"));
    final Record out = _engine.callScript("testAttachments.deleteAttachment", in);
    assertFalse(out.hasAttachment("Content"));
  }

}
