/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.bpel.pipelet.ErrorPipelet;
import org.eclipse.smila.processing.bpel.pipelet.ErrorPipelet.ErrorMode;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

/**
 * Test execution of pipelets within scripts.
 */
public class TestCallingPipeletsInScripts {

  private ScriptingEngine _scriptEngine;

  @Before
  public void setup() throws Exception {
    _scriptEngine = ServiceUtils.getService(ScriptingEngine.class);
  }

  @Test
  /** test where pipelet is created outside function. */
  public void testPipeletCreatedOutside() throws Exception {
    final String recordId = "id1";
    final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testPipeletCreatedOutside", inputRecord);
    assertNotNull(resultRecord);
    assertEquals(recordId, resultRecord.getId());
    assertEquals("testValue", resultRecord.getMetadata().getStringValue("testAtt"));
  }

  @Test
  /** test where script using pipelet returns no result. */
  public void testScriptReturnsNoResult() throws Exception {
    final String recordId = "id2";
    final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testScriptReturnsNoResult", inputRecord);
    assertNull(resultRecord);
    assertEquals("testValue", inputRecord.getMetadata().getStringValue("testAtt"));
  }

  @Test
  /** test where pipelet is created inside function. */
  public void testPipeletCreatedInFunction() throws Exception {
    {
      final String recordId = "id3";
      final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
      final Record resultRecord =
        _scriptEngine.callScript("testPipelets.testPipeletCreatedInFunction", inputRecord);
      assertEquals(recordId, resultRecord.getId());
      assertEquals("testValue3a", resultRecord.getMetadata().getStringValue("testAtt3"));
    }
    {
      final String recordId = "id3";
      final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
      final Record resultRecord =
        _scriptEngine.callScript("testPipelets.testPipeletCreatedInFunctionReturningResult", inputRecord);
      assertEquals(recordId, resultRecord.getId());
      assertEquals("testValue3b", resultRecord.getMetadata().getStringValue("testAtt3"));
    }
  }

  @Test
  /** test where multiple pipelets are used, one uses the result of the other. */
  public void testMultiplePipelets() throws Exception {
    final String recordId = "id4";
    final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testMultiplePipelets", inputRecord);
    assertEquals(recordId, resultRecord.getId());
    assertEquals("testValue4-1", resultRecord.getMetadata().getStringValue("testAtt4-1"));
    assertEquals("testValue4-2", resultRecord.getMetadata().getStringValue("testAtt4-2"));
  }

  @Test
  /** test where a JS map is used as pipelet input. */
  public void testMapAsInput() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord();
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testMapAsInput", inputRecord);
    assertEquals("id5", resultRecord.getId());
    assertEquals("testValue", resultRecord.getMetadata().getStringValue("testAtt"));
  }

  @Test
  /** test where a pipelet creates multiple records. */
  public void testPipeletCreatesRecords() throws Exception {
    final String recordId = "id6";
    final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
    final AnyMap part1 = DataFactory.DEFAULT.createAnyMap();
    part1.put("page", 17);
    final AnyMap part2 = DataFactory.DEFAULT.createAnyMap();
    part2.put("page", 18);
    inputRecord.getMetadata().getSeq("parts", true).add(part1);
    inputRecord.getMetadata().getSeq("parts", true).add(part2);

    final Record resultRecord = _scriptEngine.callScript("testPipelets.testPipeletCreatesRecords", inputRecord);
    final AnySeq resultParts = resultRecord.getMetadata().getSeq("result");
    assertEquals(2, resultParts.size());
    assertTrue(resultParts.getMap(0).get("_recordid") != resultParts.getMap(1).get("_recordid"));
    assertEquals("id6", resultParts.getMap(0).getStringValue("_documentId"));
    assertEquals("id6", resultParts.getMap(1).getStringValue("_documentId"));
    assertEquals("for", resultParts.getMap(0).getStringValue("loop"));
    assertEquals("for", resultParts.getMap(1).getStringValue("loop"));
    assertTrue(resultParts.getMap(0).getLongValue("page") != resultParts.getMap(1).getLongValue("page"));
    assertTrue(resultParts.getMap(0).getLongValue("page") == 17 || resultParts.getMap(0).getLongValue("page") == 18);
    assertTrue(resultParts.getMap(1).getLongValue("page") == 17 || resultParts.getMap(1).getLongValue("page") == 18);
  }

  @Test
  /** test where a pipelet creates multiple records used by another pipelet as input. */
  public void testPipeletCreatesRecordsUsedAsPipeletInput() throws Exception {
    final String recordId = "id7";
    final Record inputRecord = DataFactory.DEFAULT.createRecord(recordId);
    final AnyMap part1 = DataFactory.DEFAULT.createAnyMap();
    part1.put("page", 17);
    final AnyMap part2 = DataFactory.DEFAULT.createAnyMap();
    part2.put("page", 18);
    inputRecord.getMetadata().getSeq("parts", true).add(part1);
    inputRecord.getMetadata().getSeq("parts", true).add(part2);

    final Record resultRecord =
      _scriptEngine.callScript("testPipelets.testPipeletCreatesRecordsUsedAsPipeletInput", inputRecord);
    final AnySeq resultParts = resultRecord.getMetadata().getSeq("result");
    assertEquals(2, resultParts.size());
    assertEquals("testValue", resultParts.getMap(0).getStringValue("testAtt"));
    assertEquals("testValue", resultParts.getMap(1).getStringValue("testAtt"));
    assertTrue(resultParts.getMap(0).getLongValue("page") != resultParts.getMap(1).getLongValue("page"));
    assertTrue(resultParts.getMap(0).getLongValue("page") == 17 || resultParts.getMap(0).getLongValue("page") == 18);
    assertTrue(resultParts.getMap(1).getLongValue("page") == 17 || resultParts.getMap(1).getLongValue("page") == 18);
  }

  @Test
  /** test where multiple JS maps are used as pipelet input. */
  public void testMultipleMapsAsInput() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord();
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testMultipleMapsAsInput", inputRecord);

    final AnySeq resultParts = resultRecord.getMetadata().getSeq("result");
    assertEquals(2, resultParts.size());
    assertEquals("testValue", resultParts.getMap(0).getStringValue("testAtt"));
    assertEquals("testValue", resultParts.getMap(1).getStringValue("testAtt"));
    assertTrue(resultParts.getMap(0).getLongValue("page") != resultParts.getMap(1).getLongValue("page"));
    assertTrue(resultParts.getMap(0).getLongValue("page") == 17 || resultParts.getMap(0).getLongValue("page") == 18);
    assertTrue(resultParts.getMap(1).getLongValue("page") == 17 || resultParts.getMap(1).getLongValue("page") == 18);
  }

  @Test
  /** test where input record is used as config for a pipelet. */
  public void testInputRecordAsConfig() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord();
    inputRecord.getMetadata().put("outputAttribute", "a1");
    inputRecord.getMetadata().put("value", "v1");
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testInputRecordAsConfig", inputRecord);
    assertEquals("v1", resultRecord.getMetadata().getStringValue("a1"));
  }

  @Test
  /** test where result of one pipelet is used as config for another pipelet. */
  public void testPipeletResultAsConfig() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord();
    final Record resultRecord = _scriptEngine.callScript("testPipelets.testPipeletResultAsConfig", inputRecord);
    assertEquals(Arrays.asList("v1", "v2"), resultRecord.getMetadata().getSeq("testConfig2").asStrings());
  }

  @Test
  public void testPipeletProcessingError() throws Exception {
    final Record inputRecord = getPipeletErrorRecord(ErrorPipelet.ErrorMode.PROCESSING);
    try {
      _scriptEngine.callScript("testPipelets.testPipeletError", inputRecord);
      fail("should not work");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause().getCause() instanceof ProcessingException);
      assertFalse(ex.isRecoverable());
    }
  }

  @Test
  public void testPipeletRecoverableProcessingError() throws Exception {
    final Record inputRecord = getPipeletErrorRecord(ErrorPipelet.ErrorMode.RECOVERABLE_PROCESSING);
    try {
      _scriptEngine.callScript("testPipelets.testPipeletError", inputRecord);
      fail("should not work");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause().getCause() instanceof ProcessingException);
      assertTrue(ex.isRecoverable());
    }
  }

  @Test
  public void testPipeletRuntimeError() throws Exception {
    final Record inputRecord = getPipeletErrorRecord(ErrorPipelet.ErrorMode.RUNTIME);
    try {
      _scriptEngine.callScript("testPipelets.testPipeletError", inputRecord);
      fail("should not work");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause().getCause() instanceof RuntimeException);
      assertFalse(ex.isRecoverable());
    }
  }

  private Record getPipeletErrorRecord(final ErrorMode errorMode) {
    final Record inputRecord = DataFactory.DEFAULT.createRecord();
    inputRecord.getMetadata().getMap(ErrorPipelet.class.getName(), true)
      .put(ErrorPipelet.NAME_MODE, errorMode.name());
    return inputRecord;
  }

}
