/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

public class TestDataWrappers {
  private ScriptingEngine _engine;

  @Before
  public void setup() throws Exception {
    _engine = ServiceUtils.getService(ScriptingEngine.class);
  }

  /** see testDataWrappers.setAttributes(): Manipulate the record directory from javascript. */
  @Test
  public void testSetters() throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    final AnyMap out = _engine.callScript("testDataWrappers.setAttributes", in);
    assertNotNull(out);
    assertEquals("a string", out.getValue("string").getObject());
    assertEquals(42, ((Number) out.getValue("integer").getObject()).intValue());
    assertEquals(3.14, (Double) out.getValue("double").getObject(), 0.001);
    assertTrue((Boolean) out.getValue("boolean").getObject());

    assertEquals("value", out.getMap("map").getValue("key").getObject());

    assertEquals("one", out.getSeq("seq").getValue(0).getObject());
    assertEquals(2, ((Number) out.getSeq("seq").getValue(1).getObject()).intValue());
    assertEquals(3.14, (Double) out.getSeq("seq").getValue(2).getObject(), 0.001);

    assertEquals("one", out.getSeq("complex").getValue(0).getObject());
    assertEquals("two", out.getSeq("complex").getMap(1).getSeq("keys").getValue(0).getObject());
    assertEquals("three", out.getSeq("complex").getMap(1).getSeq("keys").getValue(1).getObject());
    assertEquals(1, ((Number) out.getSeq("complex").getSeq(2).getMap(0).getValue("map").getObject()).intValue());
    assertEquals(2, ((Number) out.getSeq("complex").getSeq(2).getMap(1).getValue("map").getObject()).intValue());
    assertEquals("done", out.getSeq("complex").getValue(3).getObject());
  }

  /** see testDataWrappers.setNullValue(): Tries to remove an attribute value. */
  @Test
  public void testSetNullValue() throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.put("test", "hello");
    in.put("test2", "world");
    final AnyMap out = _engine.callScript("testDataWrappers.setNullValue", in);
    assertNull(out.get("test"));
    assertEquals("world", out.getStringValue("test2"));
  }

  /** see testDataWrappers.mapIteration(): Check if a for..in loop on an AnyMap works. */
  @Test
  public void testMapIteration() throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.put("a", 1);
    in.put("b", 2);
    in.put("c", 3);
    final AnyMap out = _engine.callScript("testDataWrappers.mapIteration", in);
    assertNotNull(out);
    assertEquals(3, out.getSeq("keys").size());
    assertEquals("a", out.getSeq("keys").getStringValue(0));
    assertEquals("b", out.getSeq("keys").getStringValue(1));
    assertEquals("c", out.getSeq("keys").getStringValue(2));
  }

  /** see testDataWrappers.seqIteration(): Check if a for..in loop on an AnySeq works. */
  @Test
  public void testSeqIteration() throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.getSeq("keys", true).add("a");
    in.getSeq("keys", true).add("b");
    in.getSeq("keys", true).add("c");
    final AnyMap out = _engine.callScript("testDataWrappers.seqIteration", in);
    assertNotNull(out);
    assertEquals(3, out.getMap("map").size());
    assertEquals("0", out.getMap("map").getStringValue("a"));
    assertEquals("1", out.getMap("map").getStringValue("b"));
    assertEquals("2", out.getMap("map").getStringValue("c"));
  }

  /** see testDataWrappers.copyValues(): Check if mixed JS/Any structures are created correctly. */
  @Test
  public void testCopyValues() throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.put("string", "value");
    in.put("number", 42);
    in.getMap("map", true).put("key", "value");
    in.getSeq("seq", true).add("element");

    // must make a copy so that original object is not changed: we need it for the assertion later.
    final AnyMap out = _engine.callScript("testDataWrappers.copyValues", DataFactory.DEFAULT.cloneAnyMap(in));
    assertNotNull(out);

    assertEquals(in, out.getMap("copy").getMap("map"));
    assertEquals(in.get("string"), out.getMap("copy").getSeq("seq").get(0));
    assertEquals(in.get("number"), out.getMap("copy").getSeq("seq").get(1));
    assertEquals(in.get("map"), out.getMap("copy").getSeq("seq").get(2));
    assertEquals(in.get("seq"), out.getMap("copy").getSeq("seq").get(3));
  }

  @Test
  public void testGetIdProperty() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord("testGetIdProperty");
    final Record out = _engine.callScript("testDataWrappers.getIdProperty", in);
    assertEquals("testGetIdProperty", out.getMetadata().getStringValue("ID"));
  }

  @Test
  public void testSetIdProperty() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("ID", "testSetIdProperty");
    final Record out = _engine.callScript("testDataWrappers.setIdProperty", in);
    assertEquals("testSetIdProperty", out.getId());
  }

  @Test
  public void testGetMetadataProperty() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("attribute", "value");
    final Record out = _engine.callScript("testDataWrappers.getMetadataProperty", in);
    assertEquals("value", out.getMetadata().getMap("metadata").getStringValue("attribute"));
  }

  @Test
  public void testDeleteAttribute() throws Exception {
    final Record in = DataFactory.DEFAULT.createRecord();
    in.getMetadata().put("attributeToKeep", "keep value");
    in.getMetadata().put("attributeToDelete", "delete value");
    final Record out = _engine.callScript("testDataWrappers.deleteAttribute", in);
    assertFalse(out.getMetadata().containsKey("attributeToDelete"));
    assertEquals("keep value", out.getMetadata().getStringValue("attributeToKeep"));
  }
}
