/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

/**
 * Test execution of pipelets within scripts.
 */
public class TestPerformance {

  private ScriptingEngine _scriptEngine;

  private WorkflowProcessor _bpelProcessor;

  private Blackboard _blackboard;

  @Before
  public void setup() throws Exception {
    _scriptEngine = ServiceUtils.getService(ScriptingEngine.class);
    _bpelProcessor = ServiceUtils.getService(WorkflowProcessor.class);
    final BlackboardFactory factory = ServiceUtils.getService(BlackboardFactory.class);
    _blackboard = factory.createTransientBlackboard();
  }

  private Record createRecord(final String id, final int numberOfLoops) {
    final Record rec = DataFactory.DEFAULT.createRecord(id);
    rec.getMetadata().put("numberOfLoops", numberOfLoops);
    rec.getMetadata().put("condition", true);
    return rec;
  }

  @Test
  public void testPerformance() throws Exception {
    final int numberOfTests = 10;
    int noOfFails = 0;
    for (int i = 0; i < numberOfTests; i++) {
      boolean success = doPerformanceTest();
      if (!success) {
        noOfFails++;
      }
    }
    // we allow one failure, this may happen under certain circumstances (GC)
    assertTrue("Failures: " + noOfFails, noOfFails <= 1);
  }

  public boolean doPerformanceTest() throws Exception {
    final int numberOfLoops = 10;
    final String recordId = "id1";

    // test with JavaScript
    Record rec = createRecord(recordId, numberOfLoops);
    final long t1 = System.nanoTime();
    Record resultRecord = _scriptEngine.callScript("testPerformance.process", rec);
    final long t2 = System.nanoTime();
    final long timeScriptingNs = t2 - t1;

    assertNotNull(resultRecord);
    assertEquals(recordId, resultRecord.getId());
    assertEquals(numberOfLoops, resultRecord.getMetadata().getSeq("attribute1").size());
    assertEquals(numberOfLoops, resultRecord.getMetadata().getSeq("attribute2").size());

    // test with BPEL
    rec = createRecord(recordId, numberOfLoops);
    _blackboard.setRecord(rec);
    final long t3 = System.nanoTime();
    final String[] resultIds = _bpelProcessor.process("testPerformance", _blackboard, new String[] { recordId });
    final long t4 = System.nanoTime();
    final long timeBpelNs = t4 - t3;

    resultRecord = _blackboard.getRecord(recordId);
    assertNotNull(resultRecord);
    assertEquals(recordId, resultRecord.getId());
    assertEquals(recordId, resultIds[0]);
    assertEquals(numberOfLoops, resultRecord.getMetadata().getSeq("attribute1").size());
    assertEquals(numberOfLoops, resultRecord.getMetadata().getSeq("attribute2").size());

    System.out.println("Time Scripting (ms): " + timeScriptingNs / (1000 * 1000));
    System.out.println("Time BPEL (ms)     : " + timeBpelNs / (1000 * 1000));

    return (timeScriptingNs < timeBpelNs);
  }

  @Test
  public void compareLoadAndExecutionTimes() throws Exception {
    final AnyMap arguments = DataFactory.DEFAULT.createAnyMap();
    final int noOfTests = 5;
    System.out.println("Testing testRequire.testAll:");
    for (int i = 0; i < noOfTests; i++) {
      compareLoadAndExecutionTimes("testRequire", "testAll", arguments, noOfTests);
    }
    System.out.println("Testing testServices.ensureStore:");
    arguments.put("store", "test-performance");
    for (int i = 0; i < noOfTests; i++) {
      compareLoadAndExecutionTimes("testServices", "ensureStore", arguments, noOfTests);
    }
    System.out.println("Testing search.process:");
    arguments.put("query", "to be or not to be");
    for (int i = 0; i < noOfTests; i++) {
      compareLoadAndExecutionTimes("application/search", "process", arguments, noOfTests);
    }
    System.out.println("Testing add.process:");
    arguments.put("Filename", "testIndexBuildAndSearch.html");
    arguments.put("MimeType", "text/html");
    arguments.put("Extension", "html");
    arguments.put("Size", "42");
    arguments.put("LastModifiedDate", DataFactory.DEFAULT.createDateTimeValue(new Date()));
    arguments
      .put("Content", "<html><head><title>Hello World</title></head> <body> Indexed by script</body></html>");
    for (int i = 0; i < noOfTests; i++) {
      compareLoadAndExecutionTimes("application/add", "process", arguments, noOfTests);
    }
  }

  public void compareLoadAndExecutionTimes(final String file, final String function, final AnyMap arguments,
    final int noOfCalls) throws Exception {
    long loadTime = 0;
    long callTime = 0;
    final long loadStart = System.nanoTime();
    try (ScriptExecutor executor = _scriptEngine.getScriptExecutor()) {
      executor.loadScript(file);
      final long loadEnd = System.nanoTime();
      loadTime += loadEnd - loadStart;
      for (int i = 0; i < noOfCalls; i++) {
        final Record in = DataFactory.DEFAULT.createRecord("id" + i);
        in.getMetadata().putAll(arguments);
        final Any content = in.getMetadata().remove("Content");
        if (content != null) {
          in.setAttachment("Content", content.toString().getBytes());
        }
        final long callStart = System.nanoTime();
        executor.call(function, in);
        final long callEnd = System.nanoTime();
        callTime += callEnd - callStart;
      }
    }

    System.out.println("Time Load (ms): " + loadTime / 1e6);
    System.out.println("Time Call (ms): " + callTime / 1e6 / noOfCalls);

  }
}
