/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptNotFoundException;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.utils.MaybeRecoverableException;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;
import org.mozilla.javascript.JavaScriptException;

public class TestScriptExecution {
  private ScriptingEngine _engine;

  public static void throwException() throws MaybeRecoverableException {
    throw new MaybeRecoverableException("not recoverable", false);
  }

  public static void throwRecoverableException() throws MaybeRecoverableException {
    throw new MaybeRecoverableException("not recoverable", true);
  }

  public static void throwRuntimeException() throws MaybeRecoverableException {
    throw new RuntimeException("runtime");
  }

  @Before
  public void setup() throws Exception {
    _engine = ServiceUtils.getService(ScriptingEngine.class);
  }

  @Test
  public void testGreetAndReturnRecord() throws Exception {
    final Record input = createNameRecord("testGreetAndReturnRecord");
    final Record output = _engine.callScript("testScriptExecution.greetAndReturnRecord", input);
    assertNotNull(output);
    assertEquals("testGreetAndReturnRecord", output.getMetadata().getStringValue("name"));
    assertEquals("Hello testGreetAndReturnRecord!", output.getMetadata().getStringValue("greetAndReturnRecord"));
  }

  @Test
  public void testGreetAndReturnNothing() throws Exception {
    final Record input = createNameRecord("testGreetAndReturnNothing");
    final Record output = _engine.callScript("testScriptExecution.greetAndReturnNothing", input);
    assertNull(output);
    assertEquals("testGreetAndReturnNothing", input.getMetadata().getStringValue("name"));
    assertEquals("Hello testGreetAndReturnNothing!", input.getMetadata().getStringValue("greetAndReturnNothing"));
  }

  @Test
  public void testGreetAndReturnNewMap() throws Exception {
    final Record input = createNameRecord("testGreetAndReturnNewMap");
    final Record output = _engine.callScript("testScriptExecution.greetAndReturnNewMap", input);
    assertNotNull(output);
    // input record unchanged
    assertEquals("testGreetAndReturnNewMap", input.getMetadata().getStringValue("name"));
    assertNull(input.getMetadata().getStringValue("greetAndReturnNewMap"));
    // but no name in output.
    assertNull(output.getMetadata().getStringValue("name"));
    assertEquals("Hello testGreetAndReturnNewMap!", output.getMetadata().getStringValue("greetAndReturnNewMap"));
  }

  @Test
  public void testGreetAndThrow() throws Exception {
    final Record input = createNameRecord("testGreetAndThrow");
    try {
      _engine.callScript("testScriptExecution.greetAndThrow", input);
      fail("expected an error");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause() instanceof JavaScriptException);
      final String throwMessage = ex.getCause().getMessage();
      assertTrue(throwMessage.contains("Hello testGreetAndThrow!")); // thrown message
      assertTrue(throwMessage.contains("testScriptExecution")); // message should contain script name
      assertTrue(throwMessage.contains("21")); // message should contain linenumber
    }
  }

  /**
   * If this test fails, it is probably because the MANIFEST.MF of the Rhino bundle does not contain a
   * "DynamicImport-Package: *" entry.
   */
  @Test
  public void testGreetWithJson() throws Exception {
    final Record input = createNameRecord("testGreetWithJson");
    final Record output = _engine.callScript("testScriptExecution.greetWithJson", input);
    assertNotNull(output);
    assertEquals("testGreetWithJson", output.getMetadata().getStringValue("name"));
    assertEquals("Hello {\"name\":\"testGreetWithJson\"}!", output.getMetadata().getStringValue("greetWithJson"));
  }

  @Test
  public void testRunMissingScript() throws Exception {
    final Record input = createNameRecord("testRunMissingScript");
    try {
      _engine.callScript("noSuchScript.noSuchFunction", input);
      fail("expected an error");
    } catch (final ScriptNotFoundException ex) {
      assertTrue(ex.getMessage().contains("noSuchScript")); // message should contain script name
    }
  }

  @Test
  public void testRunMissingFunction() throws Exception {
    final Record input = createNameRecord("testRunMissingFunction");
    try {
      _engine.callScript("testScriptExecution.noSuchFunction", input);
      fail("expected an error");
    } catch (final ScriptNotFoundException ex) {
      assertTrue(ex.getMessage().contains("testScriptExecution")); // message should contain script name
      assertTrue(ex.getMessage().contains("noSuchFunction")); // message should contain function name
    }
  }

  @Test
  public void testThrowException() throws Exception {
    final Record input = createNameRecord("testThrowException");
    try {
      _engine.callScript("testScriptExecution.throwException", input);
      fail("expected an error");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause() instanceof MaybeRecoverableException);
      assertFalse(ex.isRecoverable());
    }
  }

  @Test
  public void testThrowRecoverableException() throws Exception {
    final Record input = createNameRecord("testThrowRecoverableException");
    try {
      _engine.callScript("testScriptExecution.throwRecoverableException", input);
      fail("expected an error");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause() instanceof MaybeRecoverableException);
      assertTrue(ex.isRecoverable());
    }
  }

  @Test
  public void testThrowRuntimeException() throws Exception {
    final Record input = createNameRecord("testThrowRuntimeException");
    try {
      _engine.callScript("testScriptExecution.throwRuntimeException", input);
      fail("expected an error");
    } catch (final ScriptingEngineException ex) {
      assertTrue(ex.getCause().getCause() instanceof RuntimeException);
      assertFalse(ex.isRecoverable());
    }
  }

  private Record createNameRecord(final String name) {
    final Record r = DataFactory.DEFAULT.createRecord();
    r.getMetadata().put("name", name);
    return r;
  }
}
