/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

/**
 * Test using "require" in scripts.
 */
public class TestScriptRequire {

  private ScriptingEngine _scriptEngine;

  @Before
  public void setup() throws Exception {
    _scriptEngine = ServiceUtils.getService(ScriptingEngine.class);
  }

  @Test
  /** test where require-script is on same (top) level than called script. */
  public void testRequireSameLevel() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord("id1");
    final Record resultRecord = _scriptEngine.callScript("testRequire.test", inputRecord);
    assertNotNull(resultRecord);
    assertEquals("id1", resultRecord.getId());
    assertTrue(resultRecord.getMetadata().getBooleanValue("require"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("require2"));
  }

  @Test
  /** test where require-script is in sub folder. */
  public void testRequireSubLevel() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord("id1");
    final Record resultRecord = _scriptEngine.callScript("testRequire.testWithUtils", inputRecord);
    assertNotNull(resultRecord);
    assertEquals("id1", resultRecord.getId());
    assertTrue(resultRecord.getMetadata().getBooleanValue("require"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("utils"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("commons"));
  }

  @Test
  /** test where called script is in sub folder. */
  public void testCalledScriptInSubFolder() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord("id1");
    final Record resultRecord = _scriptEngine.callScript("test/testRequireSub.testWithUtils", inputRecord);
    assertNotNull(resultRecord);
    assertEquals("id1", resultRecord.getId());
    assertTrue(resultRecord.getMetadata().getBooleanValue("require"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("utils"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("commons"));
  }

  @Test
  /** test where called script function is exported function. */
  public void testCallExportedFunction() throws Exception {
    final Record inputRecord = DataFactory.DEFAULT.createRecord("id1");
    final Record resultRecord = _scriptEngine.callScript("utils/testUtils.test", inputRecord);
    assertNotNull(resultRecord);
    assertEquals("id1", resultRecord.getId());
    assertTrue(resultRecord.getMetadata().getBooleanValue("utils"));
    assertTrue(resultRecord.getMetadata().getBooleanValue("commons"));
  }

  @Test
  public void testRequireMoreThanOnce() throws Exception {
    final AnyMap in1 = DataFactory.DEFAULT.createAnyMap();
    final AnyMap out1 = _scriptEngine.callScript("testRequireMultiple.getCounters", in1);
    assertNotNull(out1);
    // two requires for same module should load only once
    assertEquals(1, out1.getLongValue("loaded").intValue());
    assertEquals(1, out1.getLongValue("called").intValue());
    assertEquals(1, out1.getLongValue("loaded2").intValue());
    assertEquals(2, out1.getLongValue("called2").intValue());
    assertTrue(out1.getBooleanValue("sameModule"));

    // nothing remains from first call.
    final AnyMap in2 = DataFactory.DEFAULT.createAnyMap();
    final AnyMap out2 = _scriptEngine.callScript("testRequireMultiple.getCounters", in2);
    assertNotNull(out2);
    assertEquals(1, out2.getLongValue("loaded").intValue());
    assertEquals(1, out2.getLongValue("called").intValue());
    assertEquals(1, out2.getLongValue("loaded2").intValue());
    assertEquals(2, out2.getLongValue("called2").intValue());
    assertTrue(out2.getBooleanValue("sameModule"));
  }

  @Test
  public void testLocalVariablesInModules() throws Exception {
    doTestLocalVariablesInModules(10);
  }

  private AnyMap doTestLocalVariablesInModules(final int expectedCallCount) throws Exception {
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.put("callCount", expectedCallCount);
    final AnyMap out = _scriptEngine.callScript("testRequireMultiple.getCounters", in);
    assertNotNull(out);
    assertEquals(1, out.getLongValue("loaded").intValue());
    assertEquals(1, out.getLongValue("loaded2").intValue());
    assertEquals(2 * expectedCallCount - 1, out.getLongValue("called").intValue());
    assertEquals(2 * expectedCallCount, out.getLongValue("called2").intValue());
    assertTrue(out.getBooleanValue("sameModule"));
    return out;
  }

  @Test
  public void testParallelExecution() throws Exception {
    final int expectedCallCount = 100000;
    final int noOfThreads = 4;
    final ExecutorService executor = Executors.newFixedThreadPool(noOfThreads);
    final AnyMap in = DataFactory.DEFAULT.createAnyMap();
    in.put("callCount", expectedCallCount);
    @SuppressWarnings("unchecked")
    final Future<AnyMap>[] futures = new Future[noOfThreads];
    for (int i = 0; i < noOfThreads; i++) {
      futures[i] = executor.submit(new Callable<AnyMap>() {
        @Override
        public AnyMap call() throws Exception {
          return doTestLocalVariablesInModules(expectedCallCount);
        }
      });
    }
    for (int i = 0; i < noOfThreads; i++) {
      try {
        System.out.println("Thread #" + i + ": " + futures[i].get());
      } catch (final ExecutionException ex) {
        throw (Exception) ex.getCause();
      }
    }
    executor.shutdownNow();
  }
}
