/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test.worker;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Collection;

import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;

public class ScriptProcessorWorkerTestBase {

  protected static final String OUTPUT_STORE = "perm";

  protected DefinitionPersistence _jobDefs;

  protected JobRunEngine _jobRunEngine;

  protected JobRunDataProvider _jobRunData;

  protected BulkbuilderService _bulkbuilder;

  protected ObjectStoreService _store;

  public ScriptProcessorWorkerTestBase() {
    super();
  }

  @Before
  public void getServices() throws Exception {
    _jobDefs = ServiceUtils.getService(DefinitionPersistence.class);
    _jobRunEngine = ServiceUtils.getService(JobRunEngine.class);
    _jobRunData = ServiceUtils.getService(JobRunDataProvider.class);
    _bulkbuilder = ServiceUtils.getService(BulkbuilderService.class);

    _store = ServiceUtils.getService(ObjectStoreService.class);
    _store.ensureStore(OUTPUT_STORE);
    _store.clearStore(OUTPUT_STORE);
  }

  protected String startJobWithOutput(final String jobName, final String scriptName, final AnyMap specialParameters)
    throws Exception {
    return startJob(jobName, "withOutput", scriptName, specialParameters);
  }

  protected String startJobWithoutOutput(final String jobName, final String scriptName,
    final AnyMap specialParameters) throws Exception {
    return startJob(jobName, "withoutOutput", scriptName, specialParameters);
  }

  protected String startJob(final String jobName, final String workflowName, final String scriptName,
    final AnyMap specialParameters) throws Exception {
    final AnyMap jobDef = createJobDef(jobName, workflowName, scriptName, specialParameters);
    _jobDefs.addJob(new JobDefinition(jobDef));
    return _jobRunEngine.startJob(jobName);
  }

  protected AnyMap createJobDef(final String jobName, final String workflowName, final String scriptName,
    final AnyMap specialParameters) {
    final AnyMap jobDef = DataFactory.DEFAULT.createAnyMap();
    jobDef.put("name", jobName);
    jobDef.put("workflow", workflowName);
    jobDef.getMap("parameters", true).put("script", scriptName);
    jobDef.getMap("parameters", true).put("tempStore", "temp");
    jobDef.getMap("parameters", true).put("store", OUTPUT_STORE);
    if (specialParameters != null) {
      jobDef.getMap("parameters", true).putAll(specialParameters);
    }
    return jobDef;
  }

  protected void waitForJobRunCompleted(final String jobName, final String jobId) throws Exception {
    final long maxWaitTimeMillis = 10000000L;
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      final Collection<String> completedIds = _jobRunData.getCompletedJobRunIds(jobName);
      if (completedIds.contains(jobId)) {
        return;
      }
      assertTrue("Waited too long for job to complete",
        System.currentTimeMillis() - millisStarted <= maxWaitTimeMillis);
      Thread.sleep(sleepTime);
    }
  }

  protected Record record(final String recordId) {
    final Record r = DataFactory.DEFAULT.createRecord(recordId);
    return r;
  }

  protected void assertNoOutputBulks() throws ObjectStoreException {
    assertEquals(0, _store.getStoreObjectInfos(OUTPUT_STORE, "output/").size());
  }

  protected BinaryObjectStreamIterator findOutputBulk() throws Exception {
    final Collection<StoreObject> objects = _store.getStoreObjectInfos(OUTPUT_STORE, "output/");
    assertEquals(1, objects.size());
    return new BinaryObjectStreamIterator(_store.readObject(OUTPUT_STORE, objects.iterator().next().getId()));
  }

  protected AnyMap getScriptWorkerCounter(final String jobName, final String jobRunId) throws Exception {
    return _jobRunData.getJobRunData(jobName, jobRunId).getMap("worker").getMap("1_scriptProcessor");
  }

  protected int getScriptsCounter(final String jobName, final String jobRunId) throws Exception {
    return getScriptWorkerCounter(jobName, jobRunId).getLongValue("function.scripts").intValue();
  }

}
