/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test.worker;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.junit.Test;

public class TestScriptProcessorWorker extends ScriptProcessorWorkerTestBase {
  @Test
  public void testWorkerWithOutput() throws Exception {
    final String jobName = "testWorkerWithOutput";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 8;
    for (int i = 0; i < expectedRecordCount; i++) {
      _bulkbuilder.addRecord(jobName, record(jobName + i));
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertTestWorkerScriptResults(outputRecord);
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testWorkerWithoutOutput() throws Exception {
    final String jobName = "testWorkerWithoutOutput";
    final String jobRunId = startJobWithoutOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 7;
    for (int i = 0; i < expectedRecordCount; i++) {
      _bulkbuilder.addRecord(jobName, record(jobName + i));
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);
    assertNoOutputBulks();

    assertEquals(expectedRecordCount, _store.getStoreObjectInfos(OUTPUT_STORE, "record/").size());

    final IpcAnyReader parser = new IpcAnyReader();
    for (int i = 0; i < expectedRecordCount; i++) {
      final byte[] bonRecord = _store.getObject(OUTPUT_STORE, "record/testWorkerWithoutOutput" + i);
      final AnyMap outputRecord = (AnyMap) parser.readBinaryObject(bonRecord);
      assertEquals(jobName + i, outputRecord.getStringValue(Record.RECORD_ID));
      assertTestWorkerScriptResults(outputRecord);
    }
  }

  @Test
  public void testWorkerOneRecordFails() throws Exception {
    final String jobName = "testWorkerOneRecordFails";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 9;
    final int expectedFailRecord = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("fail", i == expectedFailRecord);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    assertEquals(expectedRecordCount - 1, _store.getStoreObjectInfos(OUTPUT_STORE, "record/").size());

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        recordCount++;
        assertFalse(outputRecord.getId().equals(jobName + expectedFailRecord));
        assertTestWorkerScriptResults(outputRecord);
      }
      assertEquals(expectedRecordCount - 1, recordCount);
    }
  }

  @Test
  public void testWorkerOneRecordSkipped() throws Exception {
    final String jobName = "testWorkerOneRecordSkipped";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 5;
    final int expectedSkipRecord = 2;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("skip", i == expectedSkipRecord);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    assertEquals(expectedRecordCount - 1, _store.getStoreObjectInfos(OUTPUT_STORE, "record/").size());

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        recordCount++;
        assertFalse(outputRecord.getId().equals(jobName + expectedSkipRecord));
        assertTestWorkerScriptResults(outputRecord);
      }
      assertEquals(expectedRecordCount - 1, recordCount);
    }
  }

  @Test
  public void testWorkerAllRecordsFail() throws Exception {
    final String jobName = "testWorkerAllRecordsFail";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("fail", true);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    assertEquals("FAILED", _jobRunData.getJobRunData(jobName, jobRunId).getStringValue("state"));
    assertNoOutputBulks();
    assertEquals(0, _store.getStoreObjectInfos(OUTPUT_STORE, "record/").size());
  }

  @Test
  public void testWorkerAllRecordsSkipped() throws Exception {
    final String jobName = "testWorkerOneRecordSkipped";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 4;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("skip", true);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    assertEquals("SUCCEEDED", _jobRunData.getJobRunData(jobName, jobRunId).getStringValue("state"));
    assertNoOutputBulks();
    assertEquals(0, _store.getStoreObjectInfos(OUTPUT_STORE, "record/").size());
  }

  private void assertTestWorkerScriptResults(final Record outputRecord) {
    assertTestWorkerScriptResults(outputRecord.getMetadata());
  }

  private void assertTestWorkerScriptResults(final AnyMap outputRecord) {
    assertTrue(outputRecord.getBooleanValue("processed"));
    assertEquals("SMILA Unit Tests", outputRecord.getStringValue("processor"));
    final AnyMap recordParameters = outputRecord.getMap("taskParameters");
    assertNotNull(recordParameters);
    assertEquals("testWorker", recordParameters.getStringValue("script"));
    assertEquals(OUTPUT_STORE, recordParameters.getStringValue("store"));
  }
}
