/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test.worker;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.junit.Test;

public class TestWorkerEmit extends ScriptProcessorWorkerTestBase {

  @Test
  public void testScriptUsingEmit() throws Exception {
    final String jobName = "testScriptWithEmit";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    final String jobRunId = startJobWithOutput(jobName, "testWorkerEmit", parameters);
    final int inputRecordCount = 3;
    for (int i = 0; i < inputRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("factor", i + 1);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    final int expectedOutputCount = 6;
    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertTrue(outputRecord.getId().startsWith(jobName));
        assertTrue(outputRecord.getMetadata().getBooleanValue("cloned"));
        assertEquals(recordCount++, outputRecord.getMetadata().getLongValue("cloneCount").intValue());
      }
      assertEquals(expectedOutputCount, recordCount);
    }
  }

  @Test
  public void testScriptUsingEmitAndReturn() throws Exception {
    final String jobName = "testScriptUsingEmitAndReturn";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    final String jobRunId = startJobWithOutput(jobName, "testWorkerEmit", parameters);
    final int inputRecordCount = 3;
    for (int i = 0; i < inputRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("factor", i);
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    final int expectedOutputCount = 4;
    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertTrue(outputRecord.getId().startsWith(jobName));
        if (recordCount == 0) {
          assertFalse(outputRecord.getMetadata().getBooleanValue("cloned"));
        } else {
          assertTrue(outputRecord.getMetadata().getBooleanValue("cloned"));
          assertEquals(recordCount - 1, outputRecord.getMetadata().getLongValue("cloneCount").intValue());
        }
        recordCount++;

      }
      assertEquals(expectedOutputCount, recordCount);
    }
  }

  @Test
  public void testScriptUsingEmitInMultipleScripts() throws Exception {
    final String jobName = "testScriptUsingEmitInMultipleScripts";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    final String jobRunId = startJobWithOutput(jobName, "testWorkerEmit", parameters);
    final int inputRecordCount = 10;
    for (int i = 0; i < inputRecordCount; i++) {
      final Record record = record(jobName + i);
      record.getMetadata().put("factor", 1);
      if (i % 2 == 1) {
        record.getMetadata().put("_script", "testWorkerEmitAlternative");
      }
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    assertEquals(2, getScriptsCounter(jobName, jobRunId));

    final int expectedOutputCount = inputRecordCount;
    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertTrue(outputRecord.getId().startsWith(jobName));
        final int expectedCloneCount = recordCount / 2;
        assertEquals(jobName + recordCount + "#0", outputRecord.getId());

        final AnyMap metadata = outputRecord.getMetadata();

        final boolean expectAlternative = recordCount % 2 == 1;
        assertTrue(metadata.getBooleanValue(expectAlternative ? "clonedAlternative" : "cloned"));
        assertNull(metadata.getBooleanValue(expectAlternative ? "cloned" : "clonedAlternative"));
        assertEquals(expectedCloneCount,
          metadata.getLongValue(expectAlternative ? "cloneCountAlternative" : "cloneCount").intValue());
        assertNull(metadata.getLongValue(expectAlternative ? "cloneCount" : "cloneCountAlternative"));

        recordCount++;
      }
      assertEquals(expectedOutputCount, recordCount);
    }
  }

}
