/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.test.worker;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.junit.Test;

public class TestWorkerParameters extends ScriptProcessorWorkerTestBase {

  @Test
  public void testParameterInitFunction() throws Exception {
    final String jobName = "testParameterInitFunction";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("initializeFunction", "alternativePrepare");
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      _bulkbuilder.addRecord(jobName, record(jobName + i));
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
        assertNull(outputRecord.getMetadata().getBooleanValue("alternativeProcessed"));
        assertEquals("alternativePrepare", outputRecord.getMetadata().getStringValue("prepare"));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testParameterFunction() throws Exception {
    final String jobName = "testParameterFunction";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("function", "alternativeProcess");
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      _bulkbuilder.addRecord(jobName, record(jobName + i));
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertNull(outputRecord.getMetadata().getBooleanValue("processed"));
        assertTrue(outputRecord.getMetadata().getBooleanValue("alternativeProcessed"));
        assertEquals("prepare", outputRecord.getMetadata().getStringValue("prepare"));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testParameterFunctionAndInitFunction() throws Exception {
    final String jobName = "testParameterFunctionAndInitFunction";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("function", "alternativeProcess");
    parameters.put("initializeFunction", "alternativePrepare");
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      _bulkbuilder.addRecord(jobName, record(jobName + i));
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertNull(outputRecord.getMetadata().getBooleanValue("processed"));
        assertTrue(outputRecord.getMetadata().getBooleanValue("alternativeProcessed"));
        assertEquals("alternativePrepare", outputRecord.getMetadata().getStringValue("prepare"));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testParameterScriptInRecord() throws Exception {
    final String jobName = "testParameterScriptInRecord";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    final int alternativeRecord = 1;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      if (i == alternativeRecord) {
        record.getMetadata().put("_script", "testWorkerAlternative");
      }
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount, outputRecord.getId());
        if (recordCount++ == alternativeRecord) {
          assertTrue(outputRecord.getMetadata().getBooleanValue("processed2"));
          assertNull(outputRecord.getMetadata().getBooleanValue("processed"));
        } else {
          assertNull(outputRecord.getMetadata().getBooleanValue("processed2"));
          assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
        }
      }
      assertEquals(expectedRecordCount, recordCount);
    }

    assertEquals(2, getScriptsCounter(jobName, jobRunId));
  }

  @Test
  public void testParameterFunctionInRecord() throws Exception {
    final String jobName = "testParameterFunctionInRecord";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("function", "processRecord");
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    final int alternativeRecord = 1;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      if (i == alternativeRecord) {
        record.getMetadata().put("_function", "alternativeProcess");
      }
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount, outputRecord.getId());
        assertEquals("prepare", outputRecord.getMetadata().getStringValue("prepare"));
        if (recordCount++ == alternativeRecord) {
          assertNull(outputRecord.getMetadata().getBooleanValue("processed"));
          assertTrue(outputRecord.getMetadata().getBooleanValue("alternativeProcessed"));
        } else {
          assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
          assertNull(outputRecord.getMetadata().getBooleanValue("alternativeProcessed"));
        }
      }
      assertEquals(expectedRecordCount, recordCount);
    }

    assertEquals(1, getScriptsCounter(jobName, jobRunId));
  }

  @Test
  public void testParameterWriteAttachmentsDefault() throws Exception {
    final String jobName = "testParameterWriteAttachmentsDefault";
    final String jobRunId = startJobWithOutput(jobName, "testWorker", null);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.setAttachment("attachment", record.getId().getBytes());
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
        assertEquals(outputRecord.getId(), new String(outputRecord.getAttachmentAsBytes("attachment")));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testParameterWriteAttachmentsFalse() throws Exception {
    final String jobName = "testParameterWriteAttachmentsFalse";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("writeAttachmentsToOutput", false);
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.setAttachment("attachment", record.getId().getBytes());
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
        assertFalse(outputRecord.hasAttachment("attachment"));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

  @Test
  public void testParameterWriteAttachmentsTrue() throws Exception {
    final String jobName = "testParameterWriteAttachmentsTrue";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("writeAttachmentsToOutput", true);
    final String jobRunId = startJobWithOutput(jobName, "testWorker", parameters);
    final int expectedRecordCount = 3;
    for (int i = 0; i < expectedRecordCount; i++) {
      final Record record = record(jobName + i);
      record.setAttachment("attachment", record.getId().getBytes());
      _bulkbuilder.addRecord(jobName, record);
    }
    _jobRunEngine.finishJob(jobName, jobRunId);
    waitForJobRunCompleted(jobName, jobRunId);

    try (BinaryObjectStreamIterator output = findOutputBulk()) {
      int recordCount = 0;
      while (output.hasNext()) {
        final Record outputRecord = output.next();
        assertEquals(jobName + recordCount++, outputRecord.getId());
        assertTrue(outputRecord.getMetadata().getBooleanValue("processed"));
        assertEquals(outputRecord.getId(), new String(outputRecord.getAttachmentAsBytes("attachment")));
      }
      assertEquals(expectedRecordCount, recordCount);
    }
  }

}
