exports.wrapRecord = wrapRecord;
exports.wrapAnyMap = wrapAnyMap;
exports.toAny = toAny;
exports.getService = getService;

importPackage(java.lang);
importPackage(org.eclipse.smila.datamodel);
importPackage(org.eclipse.smila.scripting.internal);

var anyClass = Class.forName("org.eclipse.smila.datamodel.Any");
var recordClass = Class.forName("org.eclipse.smila.datamodel.Record");

/**
 * Wraps a native Java Record into an RecordWrapper object which provides a seamless interface to use the object via standard JS syntax.
 * 
 * @param {Record}
 *            a native Java Record
 * @returns {Record} wrapped for JS access.
 */
function wrapRecord(jRecord) {
  if (!jRecord) {
    throw Error("Argument must be provided.");
  }
  return new RecordWrapper(jRecord, this);
}

/**
 * Wraps a native Java AnyMap into an AnyMapWrapper object which provides a seamless interface to use the object via standard JS syntax.
 * 
 * @param {AnyMap}
 *            a native Java AnyMap
 * @returns {AnyMap} wrapped for JS access.
 */
function wrapAnyMap(jAnyMap) {
  if (!jAnyMap) {
    throw Error("Argument must be provided.");
  }
  return new AnyMapWrapper(jAnyMap, this);
}

function wrapAny(any) {
  return AnyWrapper.wrapAny(any, this);
}

/**
 * Converts given object to a corresponding native SMILA Any object. For example,
 * 
 * <pre>
 * toAny({
 *   key : &quot;value&quot;
 * })
 * </pre>
 * 
 * will create a SMILA AnyMap object that can be passed to Java functions expecting AnyMap arguments.
 * 
 * <tt>null</tt> or <tt>Undefined</tt> arguments will be converted to <tt>null</tt>.
 * 
 * @param {*}
 *            object to convert.
 * @return {AnyMap|AnySeq|Value} the converted object.
 */
function toAny(object) {
  return AnyWrapper.unwrapAny(object);
}

function createRecord(object) {
  return DataFactory.DEFAULT.createRecord(toAny(object));
}

/**
 * Gets OSGi service and wrap service methods such that they easier to use from Javascript: arguments for parameters of type Any are
 * converted or unwrapped, Any or Record results are wrapped for easier JS access.
 * 
 * This is work in progress and experimental. Feel free to use and improve ;-)
 * 
 * @param {string}
 *            serviceInterface fully qualified service interface class name.
 * @returns {object} wrapped service.
 */
function getService(serviceInterface) {
  var service = services.find(serviceInterface);
  var serviceClass = Class.forName(serviceInterface);
  var methods = serviceClass.getDeclaredMethods();
  var serviceWrapper = {};
  for (var i in methods) {
    var method = methods[i];
    serviceWrapper[method.getName()] = wrapServiceMethod(service, method);
  }
  return serviceWrapper;
}

// TODO handle overloaded methods (at least: methods with same name, but different arities)
function wrapServiceMethod(service, method) {
  var arity = method.getParameterTypes().length
  var parameterWrappers = createParameterWrappers(method.getParameterTypes());
  var resultWrapper = createResultWrapper(method.getReturnType());
  return function() {
    var javaArgs = java.lang.reflect.Array.newInstance(java.lang.Object, arity);
    for (var i=0; i<arity; i++) {
      javaArgs[i] = parameterWrappers[i](arguments[i]);
    }
    return resultWrapper(method.invoke(service, javaArgs));
  }
}

function createParameterWrappers(parameterTypes) {
  var parameterWrappers = [];
  for (var i in parameterTypes) {
    var wrapper;
    if (anyClass.isAssignableFrom(parameterTypes[i])) {
      wrapper = toAny;
    } else if (recordClass.isAssignableFrom(parameterTypes[i])) {
      wrapper = createRecord;
    } else {
      wrapper = identity;
    }
    parameterWrappers.push(wrapper);
  }
  return parameterWrappers;
}

function createResultWrapper(returnType) {
  var resultWrapper;
  if (anyClass.isAssignableFrom(returnType)) {
    resultWrapper = wrapAny;
  } else if (recordClass.isAssignableFrom(returnType)) {
    resultWrapper = wrapRecord;
  } else {
    resultWrapper = identity;
  }
  return resultWrapper;
}

function identity(argument) {
  return argument;
}

function testGetService(record) {
  var service = getService("org.eclipse.smila.search.api.SearchService");
  var result = service.searchWithScript("search.process", {
    "query" : "*:*"
  });
  print(result.$metadata.count);
  return result;
}