/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Undefined;
import org.mozilla.javascript.Wrapper;

/**
 * Utility class to help with wrapping and unwrapping of SMILA objects.
 */
public final class AnyWrapper {
  private AnyWrapper() {
    throw new UnsupportedOperationException("AnyWrapper must not be instantiated.");
  }

  /**
   * Returns an object suitable as a native Javascript object from given Any object.
   *
   * @param object
   *          an Any object, may be null.
   * @return a wrapper, if the Any object was a seq or map object, the Java value object if the Any was a value object,
   *         or null, if null was given.
   */
  public static Object wrapAny(final Any object, final Scriptable scope) {
    if (object == null) {
      return null;
    }
    if (object.isSeq()) {
      return new AnySeqWrapper((AnySeq) object, scope);
    }
    if (object.isMap()) {
      return new AnyMapWrapper((AnyMap) object, scope);
    }
    return ((Value) object).getObject();
  }

  /**
   * convert given object from a Javascript process.
   *
   * @param object
   *          Javascript object, wrapper or native. May be null or {@link org.mozilla.javascript.Undefined}.
   * @return the wrapped object if the object was a wrapper. CharSequences (e.g.
   *         {@link org.mozilla.javascript.ConsString} are converted to Any Value objects. Returns null, if the input
   *         was null or {@link org.mozilla.javascript.Undefined}. Tries to convert all other objects using
   *         {@link org.eclipse.smila.datamodel.util.AnyUtil#objectToAny}, which can throw an
   *         {@link org.eclipse.smila.datamodel.InvalidValueTypeException} if conversion is not possible.
   */
  public static Any unwrapAny(Object object) {
    if (object == null || object instanceof Undefined) {
      return null;
    }
    while (object != null && object instanceof Wrapper) {
      object = ((Wrapper) object).unwrap();
    }
    if (object instanceof Any) {
      return (Any) object;
    }
    if (object instanceof CharSequence) {
      // e.g. org.mozilla.javascript.ConsString.
      // Would be handled by AnyUtil, too ... here for optimization for a frequent case.
      return DataFactory.DEFAULT.createStringValue(object.toString());
    }
    return AnyUtil.objectToAny(object);
  }

}
