/**********************************************************************************************************************
 * Copyright (c) 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.search.api.test;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlConstants;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.SearchResultConstants;
import org.eclipse.smila.search.api.SearchService;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.search.api.helper.ResultAccessor;
import org.eclipse.smila.search.api.helper.ResultItemAccessor;
import org.eclipse.smila.search.api.pipelet.TestSearchPipelet;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Tests searching via SearchService with a Javascript search script.
 */
public class TestSearchWithScript extends DeclarativeServiceTestCase {

  /** name of test Javascript. */
  public static final String SCRIPT_NAME = "searchtest.process";

  /** name of attribute set by test script in query record. */
  public static final String ATTRIBUTE_QUERY = TestSearchPipelet.QUERY;

  /** name of attribute set by test script in result records. */
  public static final String ATTRIBUTE_TITLE = TestSearchPipelet.TITLE;

  /** search service. */
  private SearchService _search;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _search = getService(SearchService.class);
    assertNotNull(_search);
  }

  /** simple test. */
  public void test() throws Exception {
    final String queryString = "test basic";
    final Record query = DataFactory.DEFAULT.createRecord();
    query.getMetadata().put(QueryConstants.QUERY, DataFactory.DEFAULT.createStringValue(queryString));
    final Record result = _search.searchWithScript(SCRIPT_NAME, query);
    assertNotNull(result);
    assertTrue(result.getId().startsWith(SCRIPT_NAME));
    assertNotNull(result.getMetadata().get(QueryConstants.QUERY));
    assertNotNull(result.getMetadata().get(SearchResultConstants.RECORDS));
    assertEquals(QueryConstants.DEFAULT_MAXCOUNT, result.getMetadata().getSeq(SearchResultConstants.RECORDS).size());
  }

  /** test with non-default size and offset. */
  public void testNonDefault() throws Exception {
    final String queryString = "test non default";
    final int mySize = 2;
    final int myOffset = 4;
    final double myThreshold = 0.5;
    final QueryBuilder queryBuilder = new QueryBuilder().setQuery(queryString);
    queryBuilder.setMaxCount(mySize).setOffset(myOffset).setThreshold(myThreshold).setIndexName("my-index");
    final Record searchResult = _search.searchWithScript(SCRIPT_NAME, queryBuilder.getQuery());
    final ResultAccessor result = new ResultAccessor(searchResult);
    assertNotNull(result);
    assertNotNull(result.getResult());
    assertTrue(result.hasQuery());
    assertTrue(result.hasRecords());
    assertEquals("my-index", result.getResult().getMetadata().getStringValue(QueryConstants.INDEXNAME));
    checkQuery(result.getResult(), queryString, myThreshold);
    checkResult(result, mySize, myOffset, myThreshold);
  }

  /** test query record terms annotation. */
  public void testTerms() throws Exception {
    final String queryString = "test some terms";
    final int mySize = 3;
    final int myOffset = 3;
    final double myThreshold = 0.1;
    final QueryBuilder queryBuilder = new QueryBuilder().setQuery(queryString);
    queryBuilder.setMaxCount(mySize).setOffset(myOffset).setThreshold(myThreshold);
    final Record searchResult = _search.searchWithScript(SCRIPT_NAME, queryBuilder.getQuery());
    final ResultAccessor result = new ResultAccessor(searchResult);
    assertNotNull(result);
    assertTrue(result.hasQuery());
    assertTrue(result.hasRecords());
  }

  /** test with attachment. */
  public void testAttachment() throws Exception {
    final String queryString = "test attachment";
    final QueryBuilder queryBuilder = new QueryBuilder().setQuery(queryString);
    queryBuilder.setAttachment("query", queryString.getBytes());
    final Record searchResult = _search.searchWithScript(SCRIPT_NAME, queryBuilder.getQuery());
    final ResultAccessor result = new ResultAccessor(searchResult);
    assertNotNull(result);
    assertTrue(result.hasQuery());
    assertTrue(result.getResult().hasAttachment("query"));
    assertEquals(queryString, new String(result.getResult().getAttachmentAsBytes("query")));
  }

  /** test search with XML results. */
  public void testXMLSearch() throws Exception {
    final String queryString = "test search with XML results";
    final QueryBuilder queryBuilder = new QueryBuilder().setQuery(queryString);
    queryBuilder.setMaxCount(QueryConstants.DEFAULT_MAXCOUNT).setOffset(QueryConstants.DEFAULT_OFFSET)
      .setThreshold(QueryConstants.DEFAULT_THRESHOLD);
    final Document result = _search.searchAsXmlWithScript(SCRIPT_NAME, queryBuilder.getQuery());
    assertNotNull(result);
    final Element elemResult = result.getDocumentElement();
    assertNotNull(elemResult);
    assertEquals(SearchService.TAG_SEARCHRESULT, elemResult.getLocalName());
    final NodeList resultChildren = elemResult.getChildNodes();
    assertEquals(2, resultChildren.getLength());
    final Node elemWorkflow = resultChildren.item(0);
    assertEquals(SearchService.TAG_SCRIPT_NAME, elemWorkflow.getLocalName());
    final Node elemRecords = resultChildren.item(1);
    assertEquals(XmlConstants.TAG_RECORD, elemRecords.getLocalName());
    final NodeList recordChildren = elemRecords.getChildNodes();
    assertTrue(recordChildren.getLength() > 0);
  }

  /** test search with XML string results. */
  public void testXMLStringSearch() throws Exception {
    final String queryString = "test search with XML results";
    final QueryBuilder queryBuilder = new QueryBuilder().setQuery(queryString);
    queryBuilder.setMaxCount(QueryConstants.DEFAULT_MAXCOUNT).setOffset(QueryConstants.DEFAULT_OFFSET)
      .setThreshold(QueryConstants.DEFAULT_THRESHOLD);
    final String result = _search.searchAsXmlStringWithScript(SCRIPT_NAME, queryBuilder.getQuery());
    assertNotNull(result);
  }

  /**
   * check attribute values, terms and facets.
   *
   * @param result
   *          search result
   * @param queryString
   *          input query string
   * @param threshold
   *          threshold (used to compute totalHits
   */
  private void checkQuery(final Record result, final String queryString, final double threshold) {
    final int expectedCount = (int) ((1 - threshold) * 100);
    assertEquals(expectedCount, result.getMetadata().getLongValue(SearchResultConstants.COUNT).intValue());
    assertEquals(TestSearchPipelet.INDEX_SIZE, result.getMetadata().getLongValue(SearchResultConstants.INDEX_SIZE)
      .intValue());
    assertTrue(result.getMetadata().containsKey(SearchResultConstants.RUNTIME));
    assertTrue(result.getMetadata().containsKey(ATTRIBUTE_QUERY));
    assertEquals(queryString, result.getMetadata().getStringValue(ATTRIBUTE_QUERY));
  }

  /**
   * simple result check.
   */
  private void checkResult(final ResultAccessor result, final int expectedResultSize, final int resultOffset,
    final double threshold) throws Exception {
    final double expectedQualityDecrement = 0.01;
    assertEquals(expectedResultSize, result.getNumberOfRecords());
    for (int i = 0; i < expectedResultSize; i++) {
      final ResultItemAccessor record = result.getResultRecord(i);
      final int position = resultOffset + i;
      final String key = TestSearchPipelet.PREFIX + position;
      assertNotNull(record);
      final Double expectedWeight = Double.valueOf(1 - position * expectedQualityDecrement);
      assertEquals(i, record.getPosition());
      assertEquals(expectedWeight, record.getWeight());
      assertTrue(record.getWeight() >= threshold);
      assertNotNull(record.getRecordId());
      assertTrue(record.getMetadata().containsKey(ATTRIBUTE_TITLE));
      assertEquals(key, record.getMetadata().getStringValue(ATTRIBUTE_TITLE));
    }
  }
}
