/**********************************************************************************************************************
 * Copyright (c) 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.search.api.helper;

import org.apache.commons.lang.NullArgumentException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.SearchResultConstants;

/**
 * Wrapper for a complete search result object. Basically provides method to create access helpers for the single query
 * and result records contained in the search result. Can also create a new query builder based on the effective query
 * object in the search result.
 */
public class ResultAccessor {

  /** the search result. */
  protected final Record _result;

  /**
   * @param result
   *          the search result.
   */
  public ResultAccessor(final Record result) {
    if (result == null) {
      throw new NullArgumentException("result");
    }
    _result = result;
  }

  /**
   * access original result.
   *
   * @return the search result.
   */
  public Record getResult() {
    return _result;
  }

  /**
   * @return true if result contains a query part.
   */
  public boolean hasQuery() {
    return _result.getMetadata().containsKey(QueryConstants.QUERY);
  }

  /**
   * @return effective query part of the result
   */
  public Any getQuery() {
    if (hasQuery()) {
      return _result.getMetadata().get(QueryConstants.QUERY);
    }
    return null;
  }

  /** get number of hits, i.e. those that match the query of which only a subset is returned. */
  public long getCount() {
    return getLongProperty(SearchResultConstants.COUNT);
  }

  /** get size of search index. */
  public long getIndexSize() {
    return getLongProperty(SearchResultConstants.INDEX_SIZE);
  }

  /** get runtime duration of search. */
  public long getRuntime() {
    return getLongProperty(SearchResultConstants.RUNTIME);
  }

  /** get facets (facets). */
  public AnyMap getFacets() {
    if (_result.getMetadata().containsKey(SearchResultConstants.FACETS)) {
      return _result.getMetadata().getMap(SearchResultConstants.FACETS);
    }
    return null;
  }

  /** get property as long value. */
  private long getLongProperty(final String name) {
    if (_result.getMetadata().containsKey(name)) {
      return _result.getMetadata().getLongValue(name);
    }
    return 0;
  }

  /**
   *
   * @return true if the result contains a (possibly empty) records list. false, if no records list is present.
   */
  public boolean hasRecords() {
    return _result.getMetadata().containsKey(SearchResultConstants.RECORDS)
      && _result.getMetadata().get(SearchResultConstants.RECORDS).isSeq();
  }

  /**
   * @return result records, if any exist. Else null.
   */
  public AnySeq getResultRecords() {
    if (hasRecords()) {
      return _result.getMetadata().getSeq(SearchResultConstants.RECORDS);
    }
    return null;
  }

  /**
   * @return number of result records.
   */
  public int getNumberOfRecords() {
    if (hasRecords()) {
      return getResultRecords().size();
    }
    return 0;
  }

  /**
   * create a wrapper for the n'th result record.
   *
   * @param index
   *          position in result list.
   * @return wrapper for the result record, or null if index in invalid.
   */
  public ResultItemAccessor getResultRecord(final int index) {
    if (index >= 0 && index < getNumberOfRecords()) {
      return new ResultItemAccessor(index, getResultRecords().getMap(index));
    }
    return null;
  }

  /**
   * @return the groups {@link AnyMap} or null if not exists.
   */
  public AnyMap getGroups() {
    return _result.getMetadata().getMap(SearchResultConstants.GROUPS);
  }

}
