/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr;

import junit.framework.TestCase;

import org.apache.commons.lang.NullArgumentException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;

public class SolrConfig_Test extends TestCase {

  private AnyMap getSampleConfig() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    config.put("key", "value");
    config.put("mode", SolrConfig.Mode.CLOUD.toString());
    final AnySeq seq = config.getSeq("seq", true);
    seq.add("seqValue");
    return config;
  }

  public void test_SolrConfig() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    assertEquals("value", config.getStringValue("key"));
    assertEquals("seqValue", config.getSeq("seq").get(0).asValue().asString());
    assertEquals(SolrConfig.Mode.CLOUD.toString(), solrConfig.getMode().toString());
  }
  
  public void test_SolrConfigModeException() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    config.put("mode", "any");
    try {
      solrConfig.getMode();
    } catch (Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_SolrConfigMode() {
    final SolrConfig solrConfig = new SolrConfig(DataFactory.DEFAULT.createAnyMap());
    try {
      solrConfig.getMode();
    } catch (final Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_getIdField() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    final AnyMap idFields = config.getMap(SolrConfig.ID_FIELDS, true);
    final String index = "index";
    final String id = "_recordid";
    idFields.put(index, id);
    assertEquals(id, solrConfig.getIdField(index));
  }

  public void test_getIdFieldDefault() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final String index = "index";
    assertEquals(SolrConfig.ID_FIELD_DEFAULT, solrConfig.getIdField(index));
  }

  public void test_getRestUri() {
    final String uri = "http://localhost:8983/solr";
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    config.put(SolrConfig.REST_URI, uri);
    assertEquals(uri, solrConfig.getRestUri());
  }

  public void test_getRestUriBlank() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    try {
      solrConfig.getRestUri();
    } catch (Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_isFetchFacetFieldType() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    config.put(SolrConfig.FETCH_FACET_FIELD_TYPE, true);
    assertTrue(solrConfig.isFetchFacetFieldType());
  }

  public void test_isFetchFacetFieldTypeNull() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    assertFalse(solrConfig.isFetchFacetFieldType());
  }

  public void test_getLoadBalancedSolrServerServerUrls() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    final AnyMap config = solrConfig.getConfigMap();
    final AnySeq serverUrls = config.getSeq(SolrConfig.SERVER_URLS, true);
    final String server1 = "http://localhost:8983/solr";
    final String server2 = "https://localhost:9983/solr";
    final String server3 = "egal";

    serverUrls.add(server1);
    serverUrls.add(server2);
    serverUrls.add(server3);

    assertEquals(server1, solrConfig.getLoadBalancedSolrServerServerUrls()[0]);
    assertEquals(server2, solrConfig.getLoadBalancedSolrServerServerUrls()[1]);
    assertEquals(server3, solrConfig.getLoadBalancedSolrServerServerUrls()[2]);
  }

  public void test_getLoadBalancedSolrServerServerUrlsNull() {
    final SolrConfig solrConfig = new SolrConfig(getSampleConfig());
    try {
      solrConfig.getLoadBalancedSolrServerServerUrls();
    } catch (Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_ZkHost() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final String testValue = "zkHost.localhost";
    config.put(SolrConfig.ZK_HOST, testValue);
    final SolrConfig solrConfig = new SolrConfig(config);
    assertEquals(testValue, solrConfig.getCloudSolrServerZkHost());
  }

  public void test_ZkHostNull() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final SolrConfig solrConfig = new SolrConfig(config);
    try {
      solrConfig.getCloudSolrServerZkHost();
    } catch (final Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  // Cloud Config

  public void test_getUpdatesToLeaders() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final boolean testValue = false;
    config.put(SolrConfig.UPDATES_TO_LEADERS, testValue);
    final SolrConfig solrConfig = new SolrConfig(config);
    assertEquals(testValue, solrConfig.getCloudSolrServerUpdatesToLeaders());
  }

  public void test_getUpdatesToLeadersDefault() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final SolrConfig solrConfig = new SolrConfig(config);
    assertEquals(SolrConfig.UPDATES_TO_LEADERS_DEFAULT, solrConfig.getCloudSolrServerUpdatesToLeaders());
  }

  // Embedded Config
  public void test_getSolrHome() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final String testValue = "localhost";
    config.put(SolrConfig.SOLR_HOME, testValue);
    final SolrConfig solrConfig = new SolrConfig(config);
    assertEquals(testValue, solrConfig.getEmbeddedSolrServerSolrHome());
  }

  public void test_getSolrHomeNull() {
    try {
      new SolrConfig(DataFactory.DEFAULT.createAnyMap()).getEmbeddedSolrServerSolrHome();
    } catch (final Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  // SolrConfig

  public void test_getBaseUrl() {
    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    final String testValue = "localhost";
    config.put(SolrConfig.BASE_URL, testValue);
    final SolrConfig solrConfig = new SolrConfig(config);
    assertEquals(testValue, solrConfig.getHttpSolrServerBaseUrl());
  }

  public void test_getBaseUrlNull() {
    try {
      new SolrConfig(DataFactory.DEFAULT.createAnyMap()).getHttpSolrServerBaseUrl();
    } catch (final Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
  }
}
