/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

import junit.framework.TestCase;

import org.apache.commons.lang.NullArgumentException;
import org.apache.solr.client.solrj.response.SolrResponseBase;
import org.apache.solr.common.util.NamedList;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;

public class SolrUtils_Test extends TestCase {

  public void test_isErrorResponse() {
    final SolrResponseBase solrResponseBase = new SolrResponseBase();
    final NamedList<Object> response = new NamedList<>();
    final NamedList<Object> responseHeader = new NamedList<>();
    responseHeader.add(SolrConstants.STATUS, 1);
    response.add(SolrConstants.RESPONSE_HEADER, responseHeader);
    solrResponseBase.setResponse(response);

    assertTrue(SolrUtils.isErrorResponse(solrResponseBase));
    responseHeader.remove(SolrConstants.STATUS);
    responseHeader.add(SolrConstants.STATUS, 0);
    assertFalse(SolrUtils.isErrorResponse(solrResponseBase));
  }

  public void test_escape() {
    assertEquals("\\+\\-\\&\\|\\!\\(\\)\\{\\}\\[\\]\\^\\~\\*\\?\\:\\\\\\\"\\/",
      SolrUtils.escape(SolrUtils.ESCAPE_CHARS));
  }

  public void test_escapeWS() {
    assertEquals("\\+\\-\\&\\|\\!\\(\\)\\{\\}\\[\\]\\^\\~\\*\\?\\:\\\\\\\"\\/\\ ",
      SolrUtils.escapeWS(SolrUtils.ESCAPE_CHARS_WS));
  }

  public void test_escape2() {
    assertEquals("\\g", SolrUtils.escape("g", "g"));
  }

  public void test_escapeEmpty() {
    assertEquals("", SolrUtils.escape("", "g"));
  }

  public void test_getPerFieldParameter() {
    final String fieldName = "fieldName";
    final String parameterName = "parameter";
    assertEquals("f." + fieldName + "." + parameterName, SolrUtils.getPerFieldParameter(fieldName, parameterName));
  }

  public void test_appendLocalParams() {
    final StringBuilder sb = new StringBuilder();
    final AnyMap localParams = DataFactory.DEFAULT.createAnyMap();
    localParams.put("key1", "val1");
    localParams.put("key2", "val2");
    localParams.put("key3", "val3");
    SolrUtils.writeLocalParams(sb, localParams);
    assertEquals("{!key1=val1 key2=val2 key3=val3}", sb.toString());
  }

  public void test_appendLocalParamsNull() {
    final StringBuilder sb = null;
    try {
      SolrUtils.writeLocalParams(sb, null);
    } catch (final Exception e) {
      if (e instanceof NullArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_parseNamedList() {
    final NamedList<Object> namedList = new NamedList<>();
    namedList.add("number", 23);
    namedList.add("string", "string");
    final NamedList<Object> namedSubList = new NamedList<>();
    namedSubList.add("subNumber", 24);
    namedSubList.add("subString", "string2");
    namedList.add("sublist", namedSubList);
    final ArrayList<Object> arraySubList = new ArrayList<>();
    arraySubList.add("arraySubString");
    arraySubList.add(27);
    namedList.add("arraySubList", arraySubList);
    assertEquals(
      "{number=23, string=string, sublist={subNumber=24, subString=string2}, arraySubList=[arraySubString, 27]}",
      SolrUtils.parseNamedList(namedList).toString());
  }

  public void test_parseNamedListNull() {
    final AnyMap target = null;
    SolrUtils.parseNamedList(null, target, null);
  }

  public void test_parseArrayList() {
    final AnySeq target = DataFactory.DEFAULT.createAnySeq();
    final ArrayList<Object> list = new ArrayList<>();
    list.add(23);
    list.add("string");
    final NamedList<Object> namedSubList = new NamedList<>();
    namedSubList.add("subNumber", 24);
    namedSubList.add("subString", "string2");
    list.add(namedSubList);
    final ArrayList<Object> arraySubList = new ArrayList<>();
    arraySubList.add("arraySubString");
    arraySubList.add(27);
    list.add(arraySubList);
    SolrUtils.parseArrayList(list, target, DataFactory.DEFAULT);
    assertEquals("[23, string, {subNumber=24, subString=string2}, [arraySubString, 27]]", target.toString());
  }

  public void test_parseArrayListNull() {
    SolrUtils.parseArrayList(null, null, null);
  }

  public void test_getSolrQueryFilter() {
    final String luceneFilter = SolrUtils.QUERY_WRAPPER_FILTER + "\\" + SolrUtils.QUERY_WRAPPER_FILTER + "blub";
    final String filtered = SolrUtils.getSolrQueryFilter(luceneFilter);
    assertEquals("\\\\blub", filtered);
  }

  public void test_getConstQueryOnfield() {
    final String fieldName = "Content";
    final String constant = "PI";
    final String result = SolrUtils.getConstQueryOnField(fieldName, constant);
    assertEquals("Content:\"PI\"", result);
  }

  public void test_getTokenQueryOnField() {
    final String fieldName = "Content";
    final String constant = "PI";
    final String result = SolrUtils.getTokenQueryOnField(fieldName, constant);
    assertEquals("(Content:PI)", result);
  }

  public void test_encodeQuery() throws UnsupportedEncodingException {
    final String query = "q=test,fl=blub";
    final String result = SolrUtils.encodeQuery(query);
    assertEquals("q%3Dtest%2Cfl%3Dblub", result);
  }

  public void test_dencodeQuery() throws UnsupportedEncodingException {
    final String query = "q%3Dtest%2Cfl%3Dblub";
    final String result = SolrUtils.decodeQuery(query);
    assertEquals("q=test,fl=blub", result);
  }

}
