/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr.params;

import java.io.IOException;

import junit.framework.TestCase;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.impl.BlackboardFactoryImpl;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;

public class SolrParams_Test extends TestCase {
  private static final String SEQ_VALUE = "seq-value";

  private static final String MAP_VALUE = "map-value";

  private static final String ANY_VALUE = "any-Value";

  private static final String ANY2 = "any";

  private static final String MAP = "map";

  private static final String SEQ = "seq";

  private static final String BOOLEAN = "boolean";

  private static final String FLOAT = "float";

  private static final String INTEGER = "integer";

  private static final String STRING = "string";

  private static final String STRING_VALUE = "string-value";

  private static final String CATEGORY = "category";

  private final DataFactory _factory = DataFactory.DEFAULT;

  private final String _recordid = "id-value";

  private AnyMap getConfigurationMap() {
    final AnyMap config = _factory.createAnyMap();
    final AnyMap category = config.getMap(CATEGORY, true);
    final Any any = _factory.createStringValue(ANY_VALUE);
    final Any anymap = _factory.createAnyMap();
    ((AnyMap) anymap).put(MAP, MAP_VALUE);
    final Any anyseq = _factory.createAnySeq();
    ((AnySeq) anyseq).add(SEQ_VALUE);
    final Any anyboolean = _factory.createBooleanValue(true);
    final Any anyfloat = _factory.createDoubleValue(23.4711f);
    final Any anyinteger = _factory.createLongValue(Integer.MAX_VALUE);
    final Any anystring = _factory.createStringValue(STRING_VALUE);
    category.put(ANY2, any);
    category.put(MAP, anymap);
    category.put(SEQ, anyseq);
    category.put(BOOLEAN, anyboolean);
    category.put(FLOAT, anyfloat);
    category.put(INTEGER, anyinteger);
    category.put(STRING, anystring);
    return config;
  }

  private Blackboard getBlackboard() {
    final Blackboard blackboard = new BlackboardFactoryImpl().createTransientBlackboard();
    final Record record = _factory.createRecord(_recordid);
    record.getMetadata().put(SolrParams.SOLR_PARAMETER_ATTRIBUTE, getConfigurationMap());
    try {
      blackboard.setRecord(record);
      blackboard.commit();
    } catch (final BlackboardAccessException e) {
      fail();
    }
    return blackboard;
  }

  public void test_getSolrParamConfigMap() throws IOException {
    final AnyMap config = getConfigurationMap();
    final SolrParams solrParams = new SolrParams(config);
    final Any any = solrParams.getSolrParam(CATEGORY, ANY2);
    assertTrue(any.asValue().asString().equals(ANY_VALUE));
  }

  public void test_getSolrParamBlackboard() {
    {
      final Blackboard blackboard = getBlackboard();
      final SolrParams solrParams = new SolrParams(blackboard);
      solrParams.setCurrentRecord(_recordid);
      final Any any = solrParams.getSolrParam(CATEGORY, ANY2);
      assertTrue(any.asValue().asString().equals(ANY_VALUE));
    }

    final Blackboard blackboard = getBlackboard();
    final SolrParams solrParams = new SolrParams(blackboard, _recordid);
    final Any any = solrParams.getSolrParam(CATEGORY, ANY2);
    assertTrue(any.asValue().asString().equals(ANY_VALUE));
  }

  public void test_getSolrParamAsAnyMap() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final AnyMap map = solrParams.getSolrParamAsAnyMap(CATEGORY, MAP);
    assertEquals(MAP_VALUE, map.getStringValue(MAP));
  }

  public void test_getSolrParamAsAnyMapBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final AnyMap map = solrParams.getSolrParamAsAnyMap(CATEGORY, MAP);
      assertEquals(MAP_VALUE, map.getStringValue(MAP));
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final AnyMap map = solrParams.getSolrParamAsAnyMap(CATEGORY, MAP);
      assertEquals(MAP_VALUE, map.getStringValue(MAP));
    }
  }

  public void test_getSolrParamsAsSeq() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final AnySeq seq = solrParams.getSolrParamAsAnySeq(CATEGORY, SEQ);
    assertEquals(SEQ_VALUE, seq.getStringValue(0));
  }

  public void test_getSolrParamsAsSeqBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final AnySeq seq = solrParams.getSolrParamAsAnySeq(CATEGORY, SEQ);
      assertEquals(SEQ_VALUE, seq.getStringValue(0));
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final AnySeq seq = solrParams.getSolrParamAsAnySeq(CATEGORY, SEQ);
      assertEquals(SEQ_VALUE, seq.getStringValue(0));
    }
  }

  public void test_getSolrParamsAsBoolean() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final Boolean bool = solrParams.getSolrParamAsBoolean(CATEGORY, BOOLEAN);
    assertTrue(bool);
  }

  public void test_getSolrParamsAsBooleanBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final Boolean bool = solrParams.getSolrParamAsBoolean(CATEGORY, BOOLEAN);
      assertTrue(bool);
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final Boolean bool = solrParams.getSolrParamAsBoolean(CATEGORY, BOOLEAN);
      assertTrue(bool);
    }
  }

  public void test_getSolrParamsAsFloat() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final Float flo = solrParams.getSolrParamAsFloat(CATEGORY, FLOAT);
    assertEquals(23.4711f, flo);
  }

  public void test_getSolrParamsAsFloatBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final Float flo = solrParams.getSolrParamAsFloat(CATEGORY, FLOAT);
      assertEquals(23.4711f, flo);
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final Float flo = solrParams.getSolrParamAsFloat(CATEGORY, FLOAT);
      assertEquals(23.4711f, flo);
    }
  }

  public void test_getSolrParamsAsInteger() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final Integer intege = solrParams.getSolrParamAsInteger(CATEGORY, INTEGER);
    assertEquals(Integer.MAX_VALUE, intege.intValue());
  }

  public void test_getSolrParamsAsIntegerBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final Integer intege = solrParams.getSolrParamAsInteger(CATEGORY, INTEGER);
      assertEquals(Integer.MAX_VALUE, intege.intValue());
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final Integer intege = solrParams.getSolrParamAsInteger(CATEGORY, INTEGER);
      assertEquals(Integer.MAX_VALUE, intege.intValue());
    }
  }

  public void test_getSolrParamsAsString() {
    final SolrParams solrParams = new SolrParams(getConfigurationMap());
    final String string = solrParams.getSolrParamAsString(CATEGORY, STRING);
    assertEquals(STRING_VALUE, string);
  }

  public void test_getSolrParamsAsStringBlackboard() {
    {
      final SolrParams solrParams = new SolrParams(getBlackboard());
      solrParams.setCurrentRecord(_recordid);
      final String string = solrParams.getSolrParamAsString(CATEGORY, STRING);
      assertEquals(STRING_VALUE, string);
    }
    {
      final SolrParams solrParams = new SolrParams(getBlackboard(), _recordid);
      final String string = solrParams.getSolrParamAsString(CATEGORY, STRING);
      assertEquals(STRING_VALUE, string);
    }
  }

  public void test_getSolrParams() throws BlackboardAccessException {
    final Record record = getBlackboard().getRecord(_recordid);
    final AnyMap solrMap = SolrParams.getSolrParams(record);
    assertNotNull(solrMap.get(CATEGORY));
  }

  public void test_getSubParameter() throws Exception {
    // create record
    final String id = "getSubParameter";
    final Record record = _factory.createRecord(id);
    record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE, true).getMap(CATEGORY, true)
      .put("param_1", "record_value_1");
    record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE, true).getMap(CATEGORY, true)
      .put("param_2", "record_value_2");
    // create blackboard
    final Blackboard blackboard = new BlackboardFactoryImpl().createTransientBlackboard();
    blackboard.setRecord(record);
    // create configuration
    final AnyMap configuration = _factory.createAnyMap();
    configuration.getMap(CATEGORY, true).put("param_1", "configuration_value_1");
    configuration.getMap(CATEGORY, true).put("param_3", "configuration_value_3");
    // create solr params
    final SolrParams params = new SolrParams(blackboard, configuration, id);
    final String param_1 = params.getSolrParamAsString(CATEGORY, "param_1");
    assertEquals("record_value_1", param_1);
    final String param_2 = params.getSolrParamAsString(CATEGORY, "param_2");
    assertEquals("record_value_2", param_2);
    final String param_3 = params.getSolrParamAsString(CATEGORY, "param_3");
    assertEquals("configuration_value_3", param_3);
    final String param_4 = params.getSolrParamAsString(CATEGORY, "param_4");
    assertNull(param_4);
  }

}
