/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr.params;

import org.apache.commons.lang.NullArgumentException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.solr.params.UpdateParams;
import org.eclipse.smila.solr.params.UpdateParams.Operation;

import junit.framework.TestCase;

public class UpdateParams_Test extends TestCase {

  private static final String QUERY_VALUE = "query-value";

  private static final String DEFAULT_CORE = "DefaultCore";

  private final DataFactory _factory = DataFactory.DEFAULT;

  private AnyMap getConfigurationMap() {
    final AnyMap configuration = _factory.createAnyMap();
    configuration.put(QueryConstants.INDEXNAME, DEFAULT_CORE);
    final AnyMap importing = configuration.getMap(UpdateParams.UPDATE, true);
    importing.put(UpdateParams.DELETE_QUERY, QUERY_VALUE);
    return configuration;
  }

  public void test_importing() {
    final AnyMap configuration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertEquals(DEFAULT_CORE, importingParams.getIndexName());
    assertEquals(QUERY_VALUE, importingParams.getDeleteQeuery());
  }

  public void test_operation() {
    final AnyMap configuration = getConfigurationMap();
    configuration.getMap(UpdateParams.UPDATE, false).put(UpdateParams.OPERATION, Operation.ADD.name());
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertEquals(Operation.ADD, importingParams.getOperation());
  }

  public void test_operationNullArgumentException() {
    final AnyMap configuration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(configuration);
    try {
      importingParams.getOperation();
    } catch (Exception e) {
      assertTrue(e instanceof NullArgumentException);
      return;
    }
    fail();
  }

  public void test_operationIllegalArgumentException() {
    final AnyMap configuration = getConfigurationMap();
    configuration.getMap(UpdateParams.UPDATE, false).put(UpdateParams.OPERATION, "value");
    final UpdateParams importingParams = new UpdateParams(configuration);
    try {
      importingParams.getOperation();
    } catch (Exception e) {
      assertTrue(e instanceof IllegalArgumentException);
      return;
    }
    fail();
  }

  public void test_attachments() {
    final AnyMap conifguration = getConfigurationMap();
    conifguration.getMap(UpdateParams.UPDATE, false).put(UpdateParams.ATTACHMENTS, true);
    final UpdateParams importingParams = new UpdateParams(conifguration);
    assertTrue(importingParams.getAttachments(false));
  }

  public void test_attachmentsDefaultIfNull() {
    final AnyMap conifguration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(conifguration);
    assertNull(importingParams.getAttachments(false));
    assertFalse(importingParams.getAttachments(true));
  }

  public void test_commitWithinMs() {
    final AnyMap conifguration = getConfigurationMap();
    conifguration.getMap(UpdateParams.UPDATE, false).put(UpdateParams.COMMIT_WITHIN_MS, 23);
    final UpdateParams importingParams = new UpdateParams(conifguration);
    assertEquals(23, (int) importingParams.getCommitWithinMs((false)));
  }

  public void test_commitWithinMsDefaultIfNull() {
    final AnyMap conifguration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(conifguration);
    assertNull(importingParams.getCommitWithinMs(false));
    assertEquals(UpdateParams.COMMIT_WITHIN_MS_DEFAULT, (int) importingParams.getCommitWithinMs((true)));
  }

  public void test_documentBoost() {
    final AnyMap configuration = getConfigurationMap();
    configuration.getMap(UpdateParams.UPDATE, false).put(UpdateParams.DOCUMENT_BOOST, 47.11f);
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertEquals(47.11f, importingParams.getDocumentBoost(false));
  }

  public void test_documentBoostDefaultIfNull() {
    final AnyMap configuration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertNull(importingParams.getDocumentBoost(false));
    assertEquals(UpdateParams.DOCUMENT_BOOST_DEFAULT, importingParams.getDocumentBoost(true));
  }

  public void test_mapping() {
    final AnyMap configuration = getConfigurationMap();
    final AnyMap mapping = configuration.getMap(UpdateParams.UPDATE, false).getMap(UpdateParams.MAPPING, true);
    mapping.put("key", "value");
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertEquals(mapping, importingParams.getMapping(false));
  }

  public void test_mappingDefaultIfNull() {
    final AnyMap configuration = getConfigurationMap();
    final UpdateParams importingParams = new UpdateParams(configuration);
    assertNull(importingParams.getMapping(false));
    assertEquals(_factory.createAnyMap(), importingParams.getMapping(true));
  }

}
