/**
 *
 */
package org.eclipse.smila.solr.query;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Date;

import junit.framework.TestCase;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.common.util.DateUtil;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.impl.BlackboardFactoryImpl;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.QueryConstants.SortOrder;
import org.eclipse.smila.solr.SolrConfig;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.SolrConstants.FacetSort;
// import org.eclipse.smila.solr.SolrConstants.ErrorHandling;
import org.eclipse.smila.solr.params.QueryParams;

/**
 * @author pwissel
 *
 */
public class QueryTransformer_Test extends TestCase {

  final DataFactory _factory = DataFactory.DEFAULT;

  final String idField = "id";

  private SolrConfig getSolrConfig() {
    final AnyMap config = _factory.createAnyMap();
    final AnyMap idFields = config.getMap(SolrConfig.ID_FIELDS, true);
    idFields.put(idField, idField);
    final SolrConfig solrConfig = new SolrConfig(config);
    return solrConfig;
  }

  public void test_addCommonQueryParameters() {
    final Record record = _factory.createRecord("addCommonQueryParameters");
    final SolrQueryBuilder SolrQueryBuilder = new SolrQueryBuilder(record);
    SolrQueryBuilder.setOffset(10);
    SolrQueryBuilder.setMaxCount(20);
    SolrQueryBuilder.addResultAttributes("fl1");
    SolrQueryBuilder.addResultAttributes("fl2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addCommonQueryParameters(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected = "start=10&rows=20&fl=fl1,fl2";
    assertEquals(expected, actual);
  }

  public void test_addCommonQueryParametersDefault() {
    final Record record = _factory.createRecord("addCommonQueryParameters");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addCommonQueryParameters(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected = "start=0&rows=10";
    assertEquals(expected, actual);
  }

  public void test_addQuery() {
    final Record record = _factory.createRecord("addQuery");
    new SolrQueryBuilder(record).setQuery("expression");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addQuery(new SolrQuery());
    final String actual = query.toString();
    final String expected = "q=expression";
    assertEquals(expected, actual);
  }

  public void test_addQueryPhrase() {
    final Record record = _factory.createRecord("addQueryPhrase");
    new SolrQueryBuilder(record).setQuery("\"expression phrase\"");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addQuery(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected = "q=\"expression phrase\"";
    assertEquals(expected, actual);
  }

  public void test_addQueryLucene() {
    final Record record = _factory.createRecord("addQueryLucene");
    new SolrQueryBuilder(record).setQuery("field1:expression1 AND field2:expression2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addQuery(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected = "q=field1:expression1 AND field2:expression2";
    assertEquals(expected, actual);
  }

  public void test_addQueryFielded() {
    final Record record = _factory.createRecord("addQueryFielded");
    final SolrQueryBuilder SolrQueryBuilder = new SolrQueryBuilder(record);
    SolrQueryBuilder.setQueryAttribute("field1", "expression1");
    SolrQueryBuilder.setQueryAttribute("field2", new Object[] { "expression21", "expression22" });
    SolrQueryBuilder.setQueryAttribute("field3", "\"expression 3 as a phrase\"");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addQuery(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected =
      "q=field1:expression1 field2:(expression21 expression22) field3:\"expression 3 as a phrase\"";
    assertEquals(expected, actual);
  }

  public void test_addQuerySeq() {
    final Record record = _factory.createRecord("addQueryFielded");
    final AnySeq querySeq = record.getMetadata().getSeq(QueryConstants.QUERY, true);
    querySeq.add("searchWord");
    querySeq.add("searchWord2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addQuery(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException exception) {
      fail(exception.getMessage());
    }
    final String expected = "q=searchWord searchWord2";
    assertEquals(expected, actual);
  }

  public void test_addNativeParams() {
    final Record record = _factory.createRecord("addNativeParams");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addNative("parameter", "val1");
    solrQueryBuilder.addNative("parameter", "val3");
    solrQueryBuilder.setNative("parameter2", "val2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addNativeParams(new SolrQuery());
    String actual = null;

    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "parameter=val1&parameter=val3&parameter2=val2";
    assertEquals(expected, actual);
  }

  public void test_addNativeParamsDefault() {
    final Record record = _factory.createRecord("addNativeParams");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addNativeParams(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "";
    assertEquals(expected, actual);
  }

  public void test_addHighlighting() {
    final Record record = _factory.createRecord("addHighlighting");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addHighlightField("hlField");
    solrQueryBuilder.addHighlightByAttribute("attrib", "hlAttrib2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addHighlighting(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "hl.fl=hlField&hl.fl=attrib&hl.fl=hlAttrib2&hl=true";
    assertEquals(expected, actual);
  }

  public void test_addHighlighting2() {
    final Record record = _factory.createRecord("addHighlighting");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addHighlightField("hlField");
    solrQueryBuilder.addHighlightByAttribute("attrib", "hlAttrib2");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addCommonQueryParameters(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "start=0&rows=10&fl=" + idField;
    assertEquals(expected, actual);
  }

  public void test_addHighlightingDefault() {
    final Record record = _factory.createRecord("addHighlighting");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addHighlighting(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "";
    assertEquals(expected, actual);
  }

  public void test_addSorting() {
    final Record record = _factory.createRecord("addSorting");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addSortBy("sortingAttribute", SortOrder.ASCENDING);
    solrQueryBuilder.addSortBy("sortingAttribute2", SortOrder.DESCENDING);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addSorting(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "sort=sortingAttribute asc,sortingAttribute2 desc";
    assertEquals(expected, actual);
  }

  public void test_addSorting2() {
    final Record record = _factory.createRecord("addSorting");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addSortBy("sortingAttribute2", SortOrder.DESCENDING);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addSorting(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "sort=sortingAttribute2 desc";
    assertEquals(expected, actual);
  }

  public void test_addSortingDefault() {
    final Record record = _factory.createRecord("addSorting");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query = new QueryTransformer(queryParams, getSolrConfig(), idField).addSorting(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "";
    assertEquals(expected, actual);
  }

  public void test_addFacettingAttribute() {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    sqb.addFacetByAttribute("attribute");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals("facet.field=attribute&facet=true", query.toString());
  }

  public void test_addFacettingAttributeSortBy() throws UnsupportedEncodingException {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    sqb.addFacetByAttribute("attribute");
    final AnySeq facetBySeq = record.getMetadata().getSeq(QueryConstants.FACETBY);
    facetBySeq.getMap(0).getMap(QueryConstants.SORTBY, true).put(QueryConstants.FACETBY_SORTCRITERION, "index");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final QueryTransformer queryTransformer = new QueryTransformer(queryParams, getSolrConfig(), idField);
    final SolrQuery solrQuery = new SolrQuery();
    queryTransformer.addFaceting(solrQuery);
    final SolrQuery query = queryTransformer.addFilters(solrQuery);
    assertEquals("f.attribute." + FacetParams.FACET_SORT + "=index&" + FacetParams.FACET_FIELD + "=attribute&"
      + FacetParams.FACET + "=true", query.toString());
  }

  public void test_addFacettingQuery() {
    // TODO see QueryTransformer Line 222
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    sqb.addFacetQuery("queryText");
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    System.out.println(query.toString());
    // assertEquals("facet.field=attribute&facet=true", query.toString());
  }

  public void test_addFacettingRangeDate() throws UnsupportedEncodingException {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    final Date start = new Date(0);
    final Date end = new Date(5000000);
    final String gap = "+1";
    sqb.addFacetByRange("range", start, end, gap);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals(
      FacetParams.FACET_RANGE + "=range&" + "f." + SolrConstants.RANGE + "." + FacetParams.FACET_RANGE_START + "="
        + URLEncoder.encode(DateUtil.getThreadLocalDateFormat().format(start), "UTF-8") + "&f."
        + SolrConstants.RANGE + "." + FacetParams.FACET_RANGE_END + "="
        + URLEncoder.encode(DateUtil.getThreadLocalDateFormat().format(end), "UTF-8") + "&f." + SolrConstants.RANGE
        + "." + FacetParams.FACET_RANGE_GAP + "=" + URLEncoder.encode(gap, "UTF-8") + "&" + FacetParams.FACET
        + "=true", query.toString());
  }

  public void test_addFacettingRangeNumber() {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    final Number start = 1.0;
    final Number end = 10.0;
    final Number gap = 2.0;
    sqb.addFacetByRange("range", start, end, gap);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals(FacetParams.FACET_RANGE + "=range&" + "f." + SolrConstants.RANGE + "."
      + FacetParams.FACET_RANGE_START + "=" + start + "&f." + SolrConstants.RANGE + "."
      + FacetParams.FACET_RANGE_END + "=" + end + "&f." + SolrConstants.RANGE + "." + FacetParams.FACET_RANGE_GAP
      + "=" + gap + "&" + FacetParams.FACET + "=true", query.toString());
  }

  public void test_addFacettingInterval() {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    final String interval = "interval";
    final String test_value1 = "part1";
    final String test_value2 = "part2";
    sqb.addFacetInterval(interval, new String[] { test_value1, test_value2 });
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals(FacetParams.FACET + "=true&" + FacetParams.FACET_INTERVAL + "=" + interval + "&f." + interval
      + "." + FacetParams.FACET_INTERVAL_SET + "=" + test_value1 + "&f." + interval + "."
      + FacetParams.FACET_INTERVAL_SET + "=" + test_value2, query.toString());
  }

  public void test_addFacettingPivot() {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    final String test_value1 = "pivot1";
    final String test_value2 = "pivot2";
    sqb.addFacetPivot(test_value1, test_value2);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals(FacetParams.FACET_FIELD + "=" + test_value1 + "&" + FacetParams.FACET_FIELD + "=" + test_value2
      + "&" + FacetParams.FACET + "=true", query.toString());
  }

  public void test_addFacettingPivot2() {
    final Record record = _factory.createRecord("addFacetting");
    final SolrQueryBuilder sqb = new SolrQueryBuilder(record);
    final String test_value1 = "pivot1";
    final String test_value2 = "pivot2";
    final String test_value3 = "pivot3";
    sqb.addFacetPivot(test_value1, test_value2);
    sqb.addFacetPivot(test_value3);
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addFaceting(new SolrQuery());
    assertEquals(FacetParams.FACET_FIELD + "=" + test_value1 + "&" + FacetParams.FACET_FIELD + "=" + test_value2
      + "&" + FacetParams.FACET_FIELD + "=" + test_value3 + "&" + FacetParams.FACET + "=true", query.toString());
  }

  public void test_addGrouping() {
    final Record record = _factory.createRecord("test_addGrouping");
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder(record);
    solrQueryBuilder.addGroupByAttribute("groupValue1");
    solrQueryBuilder.addGroupByFunctionQuery("functionQuery");
    solrQueryBuilder.addGroupByQuery("query");
    System.out.println(XmlSerializationUtils.serialize2string(record));
    final QueryParams queryParams = new QueryParams(null, record.getMetadata(), null);
    final SolrQuery query =
      new QueryTransformer(queryParams, getSolrConfig(), idField).addGrouping(new SolrQuery());
    String actual = null;
    try {
      actual = URLDecoder.decode(query.toString(), "UTF-8");
    } catch (final UnsupportedEncodingException e) {
      fail(e.getMessage());
    }
    final String expected = "group=true&group.field=groupValue1&group.func=functionQuery&group.query=query";
    assertEquals(expected, actual);
  }

  public void test_FilterAllOf() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterAllOf");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    final AnySeq allOf = filter.getSeq(QueryConstants.FILTER_ALLOF, true);
    allOf.add("f1");
    allOf.add("f2");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=+(field:f1)+(field:f2)";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterNoneOf() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterNoneOf");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    final AnySeq allOf = filter.getSeq(QueryConstants.FILTER_NONEOF, true);
    allOf.add("f1");
    allOf.add("f2");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=-(field:f1)-(field:f2)";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterOneOf() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterOneOf");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    final AnySeq allOf = filter.getSeq(QueryConstants.FILTER_ONEOF, true);
    allOf.add("f1");
    allOf.add("f2");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq= (field:f1) (field:f2)";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterAtLeast() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterAtLeast");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    filter.put(QueryConstants.FILTER_ATLEAST, "least");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=+(field:[least TO *])";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterAtMost() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterAtMost");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    filter.put(QueryConstants.FILTER_ATMOST, "most");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=+(field:[* TO most])";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterGreaterThan() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterGreaterThan");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    filter.put(QueryConstants.FILTER_GREATERTHAN, "greater");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=-(field:greater)+(field:[greater TO *])";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterLowerThan() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterLowerThan");
    final AnyMap filter = _factory.createAnyMap();
    filter.put(QueryConstants.ATTRIBUTE, "field");
    filter.put(QueryConstants.FILTER_LESSTHAN, "lower");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=-(field:lower)+(field:[* TO lower])";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FilterMultiple() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterMultiple");
    final AnyMap filter1 = _factory.createAnyMap();
    filter1.put(QueryConstants.ATTRIBUTE, "field1");
    final AnySeq allOf = filter1.getSeq(QueryConstants.FILTER_ONEOF, true);
    allOf.add("f1");
    allOf.add("f2");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter1);
    final AnyMap filter2 = _factory.createAnyMap();
    filter2.put(QueryConstants.ATTRIBUTE, "field2");
    filter2.put(QueryConstants.FILTER_ATMOST, "most");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter2);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=+(field2:[* TO most])&fq= (field1:f1) (field1:f2)";
    final String expectedAlternative = "fq= (field1:f1) (field1:f2)&fq=+(field2:[* TO most])";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertTrue(expected.equals(actual) || expectedAlternative.equals(actual));
  }

  public void test_FilterGroup() throws Exception {
    // prepare record
    final Record record = _factory.createRecord("test_FilterGroup");
    final AnyMap filter1 = _factory.createAnyMap();
    filter1.put(QueryConstants.ATTRIBUTE, "field1");
    final AnySeq allOf = filter1.getSeq(QueryConstants.FILTER_ONEOF, true);
    allOf.add("f1");
    allOf.add("f2");
    filter1.put(SolrConstants.GROUP, "group");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter1);
    final AnyMap filter2 = _factory.createAnyMap();
    filter2.put(QueryConstants.ATTRIBUTE, "field2");
    filter2.put(QueryConstants.FILTER_ATMOST, "most");
    filter2.put(SolrConstants.GROUP, "group");
    record.getMetadata().getSeq(QueryConstants.FILTER, true).add(filter2);
    final QueryTransformer transformer = getQueryTransformer(record);
    final String query = transformer.addFilters(new SolrQuery()).toString();
    final String expected = "fq=+( (field1:f1) (field1:f2))+(+(field2:[* TO most]))";
    final String actual = URLDecoder.decode(query, "UTF-8");
    assertEquals(expected, actual);
  }

  public void test_FacetAttribute() throws Exception {
    final Record record = _factory.createRecord("test_FacetAttribute");
    final AnyMap facet = _factory.createAnyMap();
    facet.put(QueryConstants.ATTRIBUTE, "field");
    record.getMetadata().getSeq(QueryConstants.FACETBY, true).add(facet);
    final String query = getQueryString(record);
    final String[] expected = { "facet=true", "facet.field=field" };
    final String[] actual = getQueryParts(query);
    assertEqualQueryParts(expected, actual);
  }

  public void test_FacetSettings() throws Exception {
    final Record record = _factory.createRecord("test_FacetSettings");
    final AnyMap facet = _factory.createAnyMap();
    facet.put(QueryConstants.ATTRIBUTE, "field");
    facet.put(QueryConstants.MAXCOUNT, 5);
    facet.getMap(QueryConstants.SORTBY, true).put(QueryConstants.FACETBY_SORTCRITERION, FacetSort.INDEX.toString());
    record.getMetadata().getSeq(QueryConstants.FACETBY, true).add(facet);
    final String query = getQueryString(record);
    final String[] expected =
      { "facet=true", "facet.field=field", "f.field.facet.limit=5", "f.field.facet.sort=index" };
    final String[] actual = getQueryParts(query);
    assertEqualQueryParts(expected, actual);
  }

  public void test_FacetLocalParams() throws Exception {
    final Record record = _factory.createRecord("test_FacetLocalParams");
    final AnyMap facet = _factory.createAnyMap();
    facet.put(QueryConstants.ATTRIBUTE, "field");
    facet.getMap(SolrConstants.LOCAL_PARAMS, true).put(QueryStringConstants.KEY, "key");
    record.getMetadata().getSeq(QueryConstants.FACETBY, true).add(facet);
    final String query = getQueryString(record);
    final String[] expected = { "facet=true", "facet.field={!key=key}field" };
    final String[] actual = getQueryParts(query);
    assertEqualQueryParts(expected, actual);
  }

  public void test_FacetFilter() throws Exception {
    final Record record = _factory.createRecord("test_FacetFilter");
    final AnyMap facet = _factory.createAnyMap();
    facet.put(QueryConstants.ATTRIBUTE, "field");
    final AnyMap filter = _factory.createAnyMap();
    filter.getSeq(QueryConstants.FILTER_ONEOF, true).add("f1");
    filter.getSeq(QueryConstants.FILTER_ONEOF, true).add("f2");
    facet.getSeq(QueryConstants.FILTER, true).add(filter);
    record.getMetadata().getSeq(QueryConstants.FACETBY, true).add(facet);
    final QueryTransformer transformer = getQueryTransformer(record);
    final SolrQuery solrQuery = transformer.addFaceting(new SolrQuery());
    final String query = transformer.addFilters(solrQuery).toString();
    final String[] expected = { "facet=true", "facet.field=field", "fq= (field:f1) (field:f2)" };
    final String[] actual = getQueryParts(query);
    assertEqualQueryParts(expected, actual);
  }

  public void test_FacetFilterMultiselect() throws Exception {
    final Record record = _factory.createRecord("test_FacetFilterMultiselect");
    final AnyMap facet = _factory.createAnyMap();
    facet.put(QueryConstants.ATTRIBUTE, "field");
    facet.put(SolrConstants.MULTISELECT, true);
    final AnyMap filter = _factory.createAnyMap();
    filter.getSeq(QueryConstants.FILTER_ONEOF, true).add("f1");
    filter.getSeq(QueryConstants.FILTER_ONEOF, true).add("f2");
    facet.getSeq(QueryConstants.FILTER, true).add(filter);
    record.getMetadata().getSeq(QueryConstants.FACETBY, true).add(facet);
    final QueryTransformer transformer = getQueryTransformer(record);
    final SolrQuery solrQuery = transformer.addFaceting(new SolrQuery());
    final String query = transformer.addFilters(solrQuery).toString();
    final String[] expected =
      { "facet=true", "facet.field={!ex=fq_field}field", "fq={!tag=fq_field} (field:f1) (field:f2)" };
    final String[] actual = getQueryParts(query);
    assertEqualQueryParts(expected, actual);
  }

  private void assertEqualQueryParts(final String[] expected, final String[] actual) {
    final String expectedString = StringUtils.join(expected, ',');
    final String actualString = StringUtils.join(actual, ',');
    String message =
      String.format("Query parts must be same length!. expected (%s): '%s'. actual (%s): '%s'.", expected.length,
        expectedString, actual.length, actualString);
    assertTrue(message, ArrayUtils.isSameLength(expected, actual));
    for (final String queryPart : expected) {
      message =
        String.format("'%s' must be part of the query. expected: '%s'. actual: '%s'.", queryPart, expectedString,
          actualString);
      assertTrue(message, ArrayUtils.indexOf(actual, queryPart) != ArrayUtils.INDEX_NOT_FOUND);
    }
  }

  private String[] getQueryParts(final String queryString) throws UnsupportedEncodingException {
    final String query = URLDecoder.decode(queryString, "UTF-8");
    return StringUtils.split(query, '&');
  }

  private String getQueryString(final Record record) throws BlackboardAccessException {
    final QueryTransformer transformer = getQueryTransformer(record);
    return transformer.addFaceting(new SolrQuery()).toString();
  }

  private QueryTransformer getQueryTransformer(final Record record) throws BlackboardAccessException {
    // prepare transformer
    final QueryParams params = getQueryParams(record);
    final SolrConfig config = getSolrConfig2();
    final String index = "test";
    return new QueryTransformer(params, config, index);
  }

  private SolrConfig getSolrConfig2() {
    // prepare config
    final AnyMap config = _factory.createAnyMap();
    config.getMap(SolrConfig.ID_FIELDS, true).put("test", "id");
    return new SolrConfig(config);
  }

  private QueryParams getQueryParams(final Record record) throws BlackboardAccessException {
    // prepare params
    final String id = record.getId();
    final Blackboard blackboard = new BlackboardFactoryImpl().createTransientBlackboard();
    blackboard.setRecord(record);
    return new QueryParams(blackboard, id);
  }

}
