/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.solr.query;

import java.util.Date;

import junit.framework.TestCase;

import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.CommonParams.EchoParamStyle;
import org.apache.solr.common.params.CursorMarkParams;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.common.params.FacetParams.FacetRangeInclude;
import org.apache.solr.common.params.FacetParams.FacetRangeOther;
import org.apache.solr.common.params.GroupParams;
import org.apache.solr.common.params.HighlightParams;
import org.apache.solr.common.params.MoreLikeThisParams;
import org.apache.solr.common.params.ShardParams;
import org.apache.solr.common.params.SpellingParams;
import org.apache.solr.common.params.TermsParams;
import org.apache.solr.search.QueryParsing;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.SolrConstants.Debug;
import org.eclipse.smila.solr.SolrConstants.FacetMethod;
import org.eclipse.smila.solr.SolrConstants.FacetSort;
import org.eclipse.smila.solr.SolrConstants.GroupFormat;
import org.eclipse.smila.solr.SolrConstants.HighlightBsType;
import org.eclipse.smila.solr.SolrConstants.MoreLikeThisInterestingTerms;
import org.eclipse.smila.solr.SolrConstants.TermsRegexFlag;
import org.eclipse.smila.solr.SolrConstants.TermsSort;
import org.eclipse.smila.solr.params.QueryParams;
import org.eclipse.smila.solr.params.SolrParams;

public class SolrQueryBuilder_Test extends TestCase {

  private final static String WORKFLOW_NAME = "workflow";

  private final String testString = "StringValueTest";

  private final String fieldName = "FieldNameTest";

  private final int testInt = 23;

  private final long testLong = 4356778126l;

  private final float testFloat = 23.76234f;

  private final boolean testBoolean = true;

  public void test_setId() {
    final String id = "recordid";
    final QueryBuilder bq = new QueryBuilder();
    bq.setId(id);
    final Record record = bq.getQuery();
    assertEquals(id, record.getId());
  }

  public void test_setQuery() {
    final Record record = new QueryBuilder().setQuery(testString).getQuery();
    assertEquals(testString, record.getMetadata().getStringValue(SolrConstants.QUERY));
  }

  public void test_setStart() {
    final Record record = new SolrQueryBuilder().setStart(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getLongValue(QueryConstants.OFFSET).intValue());
  }

  public void test_setRows() {
    final Record record = new SolrQueryBuilder().setRows(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getLongValue(QueryConstants.MAXCOUNT).intValue());
  }

  public void test_addFilterQuery() {
    final Record record = new SolrQueryBuilder().addFilterQuery(testString).getQuery();

    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getSeq(CommonParams.FQ).get(0).asValue().asString());
  }

  public void test_addReturnField() {
    final Record record = new SolrQueryBuilder().addReturnField(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getSeq(CommonParams.FL).get(0).asValue().asString());
  }

  public void test_addReturnFieldAlias() {
    final Record record = new SolrQueryBuilder().addReturnField(testString, fieldName).getQuery();
    assertEquals(fieldName + ":" + testString, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getSeq(CommonParams.FL).get(0).asValue().asString());
  }

  public void test_setDebug() {
    final Debug results = Debug.RESULTS;
    final Record record = new SolrQueryBuilder().setDebug(results).getQuery();
    assertEquals(results.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(CommonParams.DEBUG));
  }

  public void test_setDebugNull() {
    try {
      new SolrQueryBuilder().setDebug(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setExplainOther() {
    final Record record = new SolrQueryBuilder().setExplainOther(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CommonParams.EXPLAIN_OTHER));
  }

  public void test_setDebugExplainStructured() {
    final Record record = new SolrQueryBuilder().setDebugExplainStructured(true).getQuery();
    assertEquals(true, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
      .getBooleanValue(CommonParams.EXPLAIN_STRUCT).booleanValue());
  }

  public void test_setDefType() {
    final Record record = new SolrQueryBuilder().setDefType(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(QueryParsing.DEFTYPE));
  }

  public void test_setTimeAllowed() {
    final Record record = new SolrQueryBuilder().setTimeAllowed(testLong).getQuery();
    assertEquals(
      testLong,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getLongValue(CommonParams.TIME_ALLOWED).longValue());
  }

  public void test_setOmitHeader() {
    final Record record = new SolrQueryBuilder().setOmitHeader(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(CommonParams.OMIT_HEADER).booleanValue());
  }

  public void test_setRequestHandler() {
    final Record record = new SolrQueryBuilder().setRequestHandler(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CommonParams.QT));
  }

  public void test_setWriterType() {
    final Record record = new SolrQueryBuilder().setWriterType(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CommonParams.WT));
  }

  // ///////////////////////////////////
  public void test_setNow() {
    final Record record = new SolrQueryBuilder().setNow(testLong).getQuery();
    assertEquals(
      testLong,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getLongValue(CommonParams.NOW).longValue());
  }

  public void test_setTimeZone() {
    final Record record = new SolrQueryBuilder().setTimezone(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CommonParams.TZ));
  }

  public void test_setEchoHandler() {
    final Record record = new SolrQueryBuilder().setEchoHandler(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(CommonParams.HEADER_ECHO_HANDLER).booleanValue());
  }

  public void test_setEchoParams() {
    final EchoParamStyle testValue = EchoParamStyle.EXPLICIT;
    final Record record = new SolrQueryBuilder().setEchoParams(testValue).getQuery();
    assertEquals(
      testValue.name(),
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CommonParams.HEADER_ECHO_PARAMS));
  }

  public void test_setEchoHandlerNull() {
    try {
      new SolrQueryBuilder().setEchoParams(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setFacet() {
    final Record record = new SolrQueryBuilder().setFacet(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(FacetParams.FACET).booleanValue());
  }

  public void test_addFacetByAttribute() {
    final Record record = new SolrQueryBuilder().addFacetByAttribute(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(QueryConstants.ATTRIBUTE));
  }

  public void test_setFacetPrefix() {
    final Record record = new SolrQueryBuilder().setFacetPrefix(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(FacetParams.FACET_PREFIX));
  }

  public void test_setFacetPrefix2() {
    final Record record = new SolrQueryBuilder().setFacetPrefix(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + FacetParams.FACET_PREFIX));
  }

  public void test_setFacetSort() {
    final FacetSort testValue = FacetSort.COUNT;
    final Record record = new SolrQueryBuilder().setFacetSort(testValue).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(FacetParams.FACET_SORT));
  }

  public void test_setFacetSort2() {
    final FacetSort testValue = FacetSort.INDEX;
    final String outputKey = "output-key";
    final Record record = new SolrQueryBuilder().setFacetSort(testValue, outputKey).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue("f." + outputKey + "." + FacetParams.FACET_SORT));
  }

  public void test_setFacetSortException() {
    try {
      new SolrQueryBuilder().setFacetSort(null, null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setFacetLimit() {
    final Record record = new SolrQueryBuilder().setFacetLimit(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_LIMIT).intValue());
  }

  public void test_setFacetLimit2() {
    final Record record = new SolrQueryBuilder().setFacetLimit(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + FacetParams.FACET_LIMIT).intValue());
  }

  public void test_setFacetOffset() {
    final Record record = new SolrQueryBuilder().setFacetOffset(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_OFFSET).intValue());
  }

  public void test_setFacetOffset2() {
    final Record record = new SolrQueryBuilder().setFacetOffset(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + FacetParams.FACET_OFFSET).intValue());
  }

  public void test_setFacetMincount() {
    final Record record = new SolrQueryBuilder().setFacetMincount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_MINCOUNT).intValue());
  }

  public void test_setFacetMincount2() {
    final Record record = new SolrQueryBuilder().setFacetMincount(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + FacetParams.FACET_MINCOUNT).intValue());
  }

  public void test_setFacetMissing() {
    final Record record = new SolrQueryBuilder().setFacetMissing(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(FacetParams.FACET_MISSING).booleanValue());
  }

  public void test_setFacetMissing2() {
    final Record record = new SolrQueryBuilder().setFacetMissing(testBoolean, fieldName).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue("f." + fieldName + "." + FacetParams.FACET_MISSING).booleanValue());
  }

  public void test_setFacetMethod() {
    final FacetMethod testValue = FacetMethod.ENUM;
    final Record record = new SolrQueryBuilder().setFacetMethod(testValue).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(FacetParams.FACET_METHOD));
  }

  public void test_setFacetMethod2() {
    final FacetMethod testValue = FacetMethod.FC;
    final Record record = new SolrQueryBuilder().setFacetMethod(testValue, fieldName).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue("f." + fieldName + "." + FacetParams.FACET_METHOD));
  }

  public void test_setFacetMethodException() {
    try {
      new SolrQueryBuilder().setFacetMethod(null, "key-output-value").getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setFacetEnumCacheMinDf() {
    final Record record = new SolrQueryBuilder().setFacetEnumCacheMinDf(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_ENUM_CACHE_MINDF).intValue());
  }

  public void test_setFacetEnumCacheMinDf2() {
    final Record record = new SolrQueryBuilder().setFacetEnumCacheMinDf(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + FacetParams.FACET_ENUM_CACHE_MINDF)
      .intValue());
  }

  public void test_setFacetThreads() {
    final Record record = new SolrQueryBuilder().setFacetThreads(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_THREADS).intValue());
  }

  public void test_addFacetByRangeNumber() {
    final Record record = new SolrQueryBuilder().addFacetByRange(testString, 1, 2, 3).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(1, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.START)
      .intValue());
    assertEquals(2, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.END)
      .intValue());
    assertEquals(3, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.GAP)
      .intValue());
  }

  public void test_addFacetByRangeNumber2() {
    final Record record = new SolrQueryBuilder().addFacetByRange(testString, 1, 2, 3, "range_number").getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(1, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.START)
      .intValue());
    assertEquals(2, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.END)
      .intValue());
    assertEquals(3, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.GAP)
      .intValue());
    assertEquals("range_number",
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(QueryStringConstants.KEY));
  }

  public void test_addFacetByRangeNumber3() {
    final Record record =
      new SolrQueryBuilder().addFacetByRange(testString, 1, 2, 3, "range_number", true).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(true,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getBooleanValue(SolrConstants.MULTISELECT)
        .booleanValue());
    assertEquals(1, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.START)
      .intValue());
    assertEquals(2, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.END)
      .intValue());
    assertEquals(3, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getLongValue(SolrConstants.GAP)
      .intValue());
    assertEquals("range_number",
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(QueryStringConstants.KEY));
  }

  public void test_addFacetByRangeDate() {
    final Date start = new Date();
    final Date end = new Date();
    final String gap = "gap";
    final Record record = new SolrQueryBuilder().addFacetByRange(testString, start, end, gap).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(start,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.START));
    assertEquals(end, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.END));
    assertEquals(gap,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.GAP));
  }

  public void test_addFacetByRangeDate2() {
    final Date start = new Date();
    final Date end = new Date();
    final String gap = "gap";
    final Record record =
      new SolrQueryBuilder().addFacetByRange(testString, start, end, gap, "range_date").getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(start,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.START));
    assertEquals(end, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.END));
    assertEquals(gap,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.GAP));
    assertEquals("range_date",
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(QueryStringConstants.KEY));
  }

  public void test_addFacetByRangeDate3() {
    final Date start = new Date();
    final Date end = new Date();
    final String gap = "gap";
    final Record record =
      new SolrQueryBuilder().addFacetByRange(testString, start, end, gap, "range_date", true).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.RANGE));
    assertEquals(true,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getBooleanValue(SolrConstants.MULTISELECT)
        .booleanValue());
    assertEquals(start,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.START));
    assertEquals(end, record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getDateValue(SolrConstants.END));
    assertEquals(gap,
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.GAP));
    assertEquals("range_date",
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(QueryStringConstants.KEY));
  }

  public void test_addFacetByRangeNumberException() {
    try {
      new SolrQueryBuilder().addFacetByRange("", 1, 2, 3).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_addFacetByRangeDateException() {
    try {
      new SolrQueryBuilder().addFacetByRange("", new Date(), new Date(), new String()).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setFacetRangeHardend() {
    final Record record = new SolrQueryBuilder().setFacetRangeHardend(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(FacetParams.FACET_RANGE_HARD_END).booleanValue());
  }

  public void test_setFacetRangeHardend2() {
    final Record record = new SolrQueryBuilder().setFacetRangeHardend(testBoolean, fieldName).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue("f." + fieldName + "." + FacetParams.FACET_RANGE_HARD_END).booleanValue());
  }

  public void test_setFacetRangeOther() {
    final FacetRangeOther testValue = FacetRangeOther.BEFORE;
    final Record record = new SolrQueryBuilder().setFacetRangeOther(testValue).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(FacetParams.FACET_RANGE_OTHER));
  }

  public void test_setFacetRangeOther2() {
    final FacetRangeOther testValue = FacetRangeOther.BETWEEN;
    final Record record = new SolrQueryBuilder().setFacetRangeOther(testValue, fieldName).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue("f." + fieldName + "." + FacetParams.FACET_RANGE_OTHER));
  }

  public void test_setFacetRangeOtherNull() {
    try {
      new SolrQueryBuilder().setFacetRangeOther(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_setFacetRangeInclude() {
    final FacetRangeInclude testValue = FacetRangeInclude.EDGE;
    final Record record = new SolrQueryBuilder().setFacetRangeInclude(testValue).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(FacetParams.FACET_RANGE_INCLUDE));
  }

  public void test_setFacetRangeInclude2() {
    final FacetRangeInclude testValue = FacetRangeInclude.LOWER;
    final Record record = new SolrQueryBuilder().setFacetRangeInclude(testValue, fieldName).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue("f." + fieldName + "." + FacetParams.FACET_RANGE_INCLUDE));
  }

  public void test_setFacetRangeIncludeNull() {
    try {
      new SolrQueryBuilder().setFacetRangeInclude(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_addFacetPivot() {
    final Record record = new SolrQueryBuilder().addFacetPivot(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).get(0).asMap().get(SolrConstants.PIVOT).asSeq()
        .getStringValue(0));
  }

  public void test_addFacetPivot2() {
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder();
    solrQueryBuilder.addFacetPivot(testString);
    solrQueryBuilder.addFacetPivot(fieldName);
    final Record record = solrQueryBuilder.getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.FACETBY).get(0).asMap().get(SolrConstants.PIVOT).asSeq()
        .getStringValue(0));
    assertEquals(fieldName,
      record.getMetadata().getSeq(QueryConstants.FACETBY).get(1).asMap().get(SolrConstants.PIVOT).asSeq()
        .getStringValue(0));
  }

  public void test_addFacetPivotNull() {
    try {
      new SolrQueryBuilder().addFacetPivot((String[]) null);
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_addFacetInterval() {
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder();
    solrQueryBuilder.addFacetInterval("interval", new String[] { "23", "24" });
    final Record record = solrQueryBuilder.getQuery();
    assertEquals("interval",
      record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getStringValue(SolrConstants.INTERVAL));
    assertEquals("23", record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getSeq(SolrConstants.SET)
      .getStringValue(0));
    assertEquals("24", record.getMetadata().getSeq(QueryConstants.FACETBY).getMap(0).getSeq(SolrConstants.SET)
      .getStringValue(1));
  }

  public void test_addFacetIntervalNull() {
    try {
      new SolrQueryBuilder().addFacetInterval("", (String[]) null);
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setFacetPivotMincount() {
    final Record record = new SolrQueryBuilder().setFacetPivotMincount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(FacetParams.FACET_PIVOT_MINCOUNT).intValue());
  }

  public void test_setGroup() {
    final Record record = new SolrQueryBuilder().setGroup(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(GroupParams.GROUP).booleanValue());
  }

  public void test_addGroupByAttribute() {
    final Record record = new SolrQueryBuilder().addGroupByAttribute(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.GROUPBY).getMap(0).getStringValue(QueryConstants.ATTRIBUTE));
  }

  public void test_addGroupByAttributeNull() {
    try {
      new SolrQueryBuilder().addGroupByAttribute(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_addGroupByFunctionQuery() {
    final Record record = new SolrQueryBuilder().addGroupByFunctionQuery(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.GROUPBY).getMap(0).getStringValue(SolrConstants.FUNC));
  }

  public void test_addGroupByFunctionQueryNull() {
    try {
      new SolrQueryBuilder().addGroupByFunctionQuery(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_addGroupByQuery() {
    final Record record = new SolrQueryBuilder().addGroupByQuery(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.GROUPBY).getMap(0).getStringValue(CommonParams.QUERY));
  }

  public void test_addGroupByQueryNull() {
    try {
      new SolrQueryBuilder().addGroupByQuery(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_setGroupLimit() {
    final Record record = new SolrQueryBuilder().setGroupLimit(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(GroupParams.GROUP_LIMIT).intValue());
  }

  public void test_setGroupOffset() {
    final Record record = new SolrQueryBuilder().setGroupOffset(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(GroupParams.GROUP_OFFSET).intValue());
  }

  public void test_setGroupFormat() {
    final GroupFormat testValue = GroupFormat.GROUPED;
    final Record record = new SolrQueryBuilder().setGroupFormat(testValue).getQuery();
    assertEquals(testValue.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(GroupParams.GROUP_FORMAT));
  }

  public void test_setGroupFormatNull() {
    try {
      new SolrQueryBuilder().setGroupFormat(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }

    }
    fail();
  }

  public void test_setGroupMain() {
    final Record record = new SolrQueryBuilder().setGroupMain(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(GroupParams.GROUP_MAIN).booleanValue());
  }

  public void test_setGroupNGroups() {
    final Record record = new SolrQueryBuilder().setGroupNGroups(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(GroupParams.GROUP + QueryStringConstants.PERIOT + SolrConstants.NGROUPS).booleanValue());
  }

  public void test_setGroupTruncate() {
    final Record record = new SolrQueryBuilder().setGroupTruncate(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(GroupParams.GROUP_TRUNCATE).booleanValue());
  }

  public void test_setGroupFacet() {
    final Record record = new SolrQueryBuilder().setGroupFacet(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(GroupParams.GROUP_FACET).booleanValue());
  }

  public void test_setGroupCachePercent() {
    final Record record = new SolrQueryBuilder().setGroupCachePercent(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(GroupParams.GROUP_CACHE_PERCENTAGE).intValue());
  }

  public void test_setGroupCachePercentFail() {
    final int testInt = -1;
    try {
      new SolrQueryBuilder().setGroupCachePercent(testInt).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setHighlight() {
    final Record record = new SolrQueryBuilder().setHighlight(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.HIGHLIGHT).booleanValue());
  }

  public void test_setHighlightQuery() {
    final Record record = new SolrQueryBuilder().setHighlightQuery(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.Q));
  }

  public void test_addHighlightField() {
    final Record record =
      new SolrQueryBuilder().addHighlightField(testString).addHighlightField(testString).getQuery();
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.HIGHLIGHT).getMap(0).getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(testString,
      record.getMetadata().getSeq(QueryConstants.HIGHLIGHT).getMap(1).getStringValue(QueryConstants.ATTRIBUTE));
  }

  public void test_addHighlightFieldFail() {
    final String testString = "";
    try {
      new SolrQueryBuilder().addHighlightField(testString).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_addHighlightFieldFail2() {
    final String testString = null;
    try {
      new SolrQueryBuilder().addHighlightField(testString).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setHighlightSnippets() {
    final Record record = new SolrQueryBuilder().setHighlightSnippets(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.SNIPPETS).intValue());
  }

  public void test_setHighlightSnippets2() {
    final String fieldName = "field-name";
    final Record record = new SolrQueryBuilder().setHighlightSnippets(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + HighlightParams.SNIPPETS).intValue());
  }

  public void test_setHighlightFragsize() {
    final Record record = new SolrQueryBuilder().setHighlightFragsize(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.FRAGSIZE).intValue());
  }

  public void test_setHighlightFragsize2() {
    final Record record = new SolrQueryBuilder().setHighlightFragsize(testInt, fieldName).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue("f." + fieldName + "." + HighlightParams.FRAGSIZE).intValue());
  }

  public void test_setHighlightMergeContiguous() {
    final Record record = new SolrQueryBuilder().setHighlightMergeContiguous(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.MERGE_CONTIGUOUS_FRAGMENTS).booleanValue());
  }

  public void test_setHighlightMergeContiguous2() {
    final Record record = new SolrQueryBuilder().setHighlightMergeContiguous(testBoolean, fieldName).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue("f." + fieldName + "." + HighlightParams.MERGE_CONTIGUOUS_FRAGMENTS).booleanValue());
  }

  public void test_setHighlightRequireFieldMatch() {
    final Record record = new SolrQueryBuilder().setHighlightRequireFieldMatch(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.FIELD_MATCH).booleanValue());
  }

  public void test_setHighlightMaxAnalysedChars() {
    final int testInt = 23;
    final Record record = new SolrQueryBuilder().setHighlightMaxAnalysedChars(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.MAX_CHARS).intValue());
  }

  public void test_setHighlightAlternateField() {
    final Record record = new SolrQueryBuilder().setHighlightAlternateField(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.ALTERNATE_FIELD));
  }

  public void test_setHighlightAlternateField2() {
    final Record record = new SolrQueryBuilder().setHighlightAlternateField(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.ALTERNATE_FIELD));
  }

  public void test_setHighlightAlternateFieldLength() {
    final Record record = new SolrQueryBuilder().setHighlightAlternateFieldLength(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.ALTERNATE_FIELD_LENGTH).intValue());
  }

  public void test_setHighlightPreserveMulti() {
    final Record record = new SolrQueryBuilder().setHighlightPreserveMulti(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.PRESERVE_MULTI).booleanValue());
  }

  public void test_setHighlightMaxMultiValuedToExamine() {
    final Record record = new SolrQueryBuilder().setHighlightMaxMultiValuedToExamine(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.MAX_MULTIVALUED_TO_EXAMINE).intValue());
  }

  public void test_setHighlightMaxMultiValuedToMatch() {
    final Record record = new SolrQueryBuilder().setHighlightMaxMultiValuedToMatch(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.MAX_MULTIVALUED_TO_MATCH).intValue());
  }

  public void test_setHighlightFormatter() {
    final Record record = new SolrQueryBuilder().setHighlightFormatter(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.FORMATTER));
  }

  public void test_setHighlightFormatter2() {
    final Record record = new SolrQueryBuilder().setHighlightFormatter(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.FORMATTER));
  }

  public void test_setHighlightSimplePre() {
    final Record record = new SolrQueryBuilder().setHighlightSimplePre(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.SIMPLE_PRE));
  }

  public void test_setHighlightSimplePre2() {
    final Record record = new SolrQueryBuilder().setHighlightSimplePre(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.SIMPLE_PRE));
  }

  public void test_setHighlightSimplePost() {
    final Record record = new SolrQueryBuilder().setHighlightSimplePost(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.SIMPLE_POST));
  }

  public void test_setHighlightSimplePost2() {
    final Record record = new SolrQueryBuilder().setHighlightSimplePost(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.SIMPLE_POST));
  }

  public void test_setHighlightFragmeter() {
    final Record record = new SolrQueryBuilder().setHighlightFragmeter(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.FRAGMENTER));
  }

  public void test_setHighlightFragmeter2() {
    final Record record = new SolrQueryBuilder().setHighlightFragmenter(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.FRAGMENTER));
  }

  public void test_setHighlightFragListBuilder() {
    final Record record = new SolrQueryBuilder().setHighlightFragListBuilder(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.FRAG_LIST_BUILDER));
  }

  public void test_setHighlightFragmentsBuilder() {
    final Record record = new SolrQueryBuilder().setHighlightFragmentsBuilder(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.FRAGMENTS_BUILDER));
  }

  public void tset_setHighlightFragListBuilder2() {
    final Record record = new SolrQueryBuilder().setHighlightFragListBuilder(testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + HighlightParams.FRAG_LIST_BUILDER));
  }

  public void test_setHighlightBoundaryScanner() {
    final Record record = new SolrQueryBuilder().setHighlightBoundaryScanner(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.BOUNDARY_SCANNER));
  }

  public void test_setHighlightBsMaxScan() {
    final Record record = new SolrQueryBuilder().setHighlightBsMaxScan(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.BS_MAX_SCAN).intValue());
  }

  public void test_setHighlightBsChars() {
    final Record record = new SolrQueryBuilder().setHighlightBsChars(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.BS_CHARS));
  }

  public void test_setHighlightBsType() {
    final HighlightBsType bsType = HighlightBsType.SENTENCE;
    final Record record = new SolrQueryBuilder().setHighlightBsType(bsType).getQuery();
    assertEquals(
      bsType.name(),
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.BS_TYPE));
  }

  public void test_setHighlightBsLanguage() {
    final Record record = new SolrQueryBuilder().setHighlightBsLanguage(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.BS_LANGUAGE));
  }

  public void test_setHighlightBsCountry() {
    final Record record = new SolrQueryBuilder().setHighlightBsCountry(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.BS_COUNTRY));
  }

  public void test_setHighlightUseFastVectorHighlighter() {
    final Record record = new SolrQueryBuilder().setHighlightUseFastVectorHighlighter(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.USE_FVH).booleanValue());
  }

  public void test_setHighlightUseFastVectorHighlighter2() {
    final Record record =
      new SolrQueryBuilder().setHighlightUseFastVectorHighlighter(testBoolean, fieldName).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue("f." + fieldName + "." + HighlightParams.USE_FVH).booleanValue());
  }

  public void test_setHighlightUsePhraseHighlighter() {
    final Record record = new SolrQueryBuilder().setHighlightUsePhraseHighlighter(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.USE_PHRASE_HIGHLIGHTER).booleanValue());
  }

  public void test_setHighlightMultiTerm() {
    final Record record = new SolrQueryBuilder().setHighlightMultiTerm(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(HighlightParams.HIGHLIGHT_MULTI_TERM).booleanValue());
  }

  public void test_setHighlightRegexSlop() {
    final Record record = new SolrQueryBuilder().setHighlightRegexSlop(testFloat).getQuery();
    assertEquals(
      testFloat,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getDoubleValue(HighlightParams.SLOP).floatValue());
  }

  public void test_setHighlightRegexPattern() {
    final Record record = new SolrQueryBuilder().setHighlightRegexPattern(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(HighlightParams.PATTERN));
  }

  public void test_setHighlightRegexMaxAnalyzedChars() {
    final Record record = new SolrQueryBuilder().setHighlightRegexMaxAnalyzedChars(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(HighlightParams.MAX_RE_CHARS).intValue());
  }

  public void test_setMoreLikeThis() {
    final Record record = new SolrQueryBuilder().setMoreLikeThis(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(MoreLikeThisParams.MLT).booleanValue());
  }

  public void test_setMoreLikeThisSimilarityFields() {
    final String[] similarityFields = new String[3];
    similarityFields[0] = testString;
    similarityFields[1] = fieldName;
    similarityFields[2] = WORKFLOW_NAME;
    final Record record = new SolrQueryBuilder().setMoreLikeThisSimilarityFields(similarityFields).getQuery();
    assertEquals(
      testString + "," + fieldName + "," + WORKFLOW_NAME,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(MoreLikeThisParams.SIMILARITY_FIELDS));
  }

  public void test_setMoreLikeThisCount() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisCount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.DOC_COUNT).intValue());
  }

  public void test_setMoreLikeThisMinTermFrequency() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMinTermFrequency(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MIN_TERM_FREQ).intValue());
  }

  public void test_setMoreLikeThisMinDocumentFrequency() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMinDocumentFrequency(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MIN_DOC_FREQ).intValue());
  }

  public void test_setMoreLikeThisMinWordLength() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMinWordLength(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MIN_WORD_LEN).intValue());
  }

  public void test_setMoreLikeThisMaxWordLength() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMaxWordLength(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MAX_WORD_LEN).intValue());
  }

  public void test_setMoreLikeThisMaxQueryTerms() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMaxQueryTerms(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MAX_QUERY_TERMS).intValue());
  }

  public void test_setMoreLikeThisMaxTokens() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMaxTokens(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MAX_NUM_TOKENS_PARSED).intValue());
  }

  public void test_setMoreLikeThisBoost() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisBoost(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(MoreLikeThisParams.BOOST).booleanValue());
  }

  public void test_setMoreLikeThisQueryFields() {
    final String[] similarityFields = new String[3];
    similarityFields[0] = testString;
    similarityFields[1] = fieldName;
    similarityFields[2] = WORKFLOW_NAME;
    final Record record = new SolrQueryBuilder().setMoreLikeThisQueryFields(similarityFields).getQuery();
    assertEquals(
      testString + " " + fieldName + " " + WORKFLOW_NAME,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(MoreLikeThisParams.QF));
  }

  public void test_setMoreLikeThisMatchInclude() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMatchInclude(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(MoreLikeThisParams.MATCH_INCLUDE).booleanValue());
  }

  public void test_setMoreLikeThisMatchOffset() {
    final Record record = new SolrQueryBuilder().setMoreLikeThisMatchOffset(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(MoreLikeThisParams.MATCH_OFFSET).intValue());
  }

  public void test_setMoreLikeThisInterestingTerms() {
    final MoreLikeThisInterestingTerms mltit = MoreLikeThisInterestingTerms.DETAILS;
    final Record record = new SolrQueryBuilder().setMoreLikeThisInterestingTerms(mltit).getQuery();
    assertEquals(mltit.name().toLowerCase(), record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getStringValue(MoreLikeThisParams.INTERESTING_TERMS));
  }

  public void test_setMoreLikeThisInterestingTermsNull() {
    try {
      new SolrQueryBuilder().setMoreLikeThisInterestingTerms(null);
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setSpellcheck() {
    final Record record = new SolrQueryBuilder().setSpellcheck(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SolrConstants.SPELLCHECK).booleanValue());
  }

  public void test_setSpellcheckQuery() {
    final Record record = new SolrQueryBuilder().setSpellcheckQuery(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SpellingParams.SPELLCHECK_Q));
  }

  public void test_setSpellcheckBuild() {
    final Record record = new SolrQueryBuilder().setSpellcheckBuild(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_BUILD).booleanValue());
  }

  public void test_setSpellcheckReload() {
    final Record record = new SolrQueryBuilder().setSpellcheckReload(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_RELOAD).booleanValue());
  }

  public void test_setSpellcheckDictionary() {
    final Record record = new SolrQueryBuilder().setSpellcheckDictionary(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SpellingParams.SPELLCHECK_DICT));
  }

  public void test_setSpellcheckCount() {
    final Record record = new SolrQueryBuilder().setSpellcheckCount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_COUNT).intValue());
  }

  public void test_setSpellcheckAlternativeTermCount() {
    final Record record = new SolrQueryBuilder().setSpellcheckAlternativeTermCount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_ALTERNATIVE_TERM_COUNT).intValue());
  }

  public void test_setSpellchekOnlyMorePopular() {
    final Record record = new SolrQueryBuilder().setSpellchekOnlyMorePopular(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_ONLY_MORE_POPULAR).booleanValue());
  }

  public void test_setSpellcheckMaxResultsForSuggest() {
    final Record record = new SolrQueryBuilder().setSpellcheckMaxResultsForSuggest(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_MAX_RESULTS_FOR_SUGGEST).intValue());
  }

  public void test_setSpellcheckExtendedResults() {
    final Record record = new SolrQueryBuilder().setSpellcheckExtendedResults(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_EXTENDED_RESULTS).booleanValue());
  }

  public void test_setSpellcheckCollate() {
    final Record record = new SolrQueryBuilder().setSpellcheckCollate(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_COLLATE).booleanValue());
  }

  public void test_setSpellcheckMaxCollations() {
    final Record record = new SolrQueryBuilder().setSpellcheckMaxCollations(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_MAX_COLLATIONS).intValue());
  }

  public void test_setSpellcheckMaxCollationTries() {
    final Record record = new SolrQueryBuilder().setSpellcheckMaxCollationTries(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_MAX_COLLATION_TRIES).intValue());
  }

  public void test_setSpellcheckMaxCollationEvaluations() {
    final Record record = new SolrQueryBuilder().setSpellcheckMaxCollationEvaluations(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_MAX_COLLATION_EVALUATIONS).intValue());
  }

  public void test_setSpellcheckCollateParam() {
    final Record record = new SolrQueryBuilder().setSpellcheckCollateParam(testString, fieldName).getQuery();
    assertEquals(
      fieldName,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SpellingParams.SPELLCHECK_COLLATE_PARAM_OVERRIDE + testString));
  }

  public void test_setSpellcheckCollateExtendedResults() {
    final Record record = new SolrQueryBuilder().setSpellcheckCollateExtendedResults(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(SpellingParams.SPELLCHECK_COLLATE_EXTENDED_RESULTS).booleanValue());
  }

  public void test_setSpellcheckCollateMaxCollectDocs() {
    final Record record = new SolrQueryBuilder().setSpellcheckCollateMaxCollectDocs(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(SpellingParams.SPELLCHECK_COLLATE_MAX_COLLECT_DOCS).intValue());
  }

  public void test_setSpellcheckAccuracy() {
    final Record record = new SolrQueryBuilder().setSpellcheckAccuracy(testFloat).getQuery();
    assertEquals(
      testFloat,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getDoubleValue(SpellingParams.SPELLCHECK_ACCURACY).floatValue());
  }

  public void test_setSpellcheckDictionaryKey() {
    final Record record =
      new SolrQueryBuilder().setSpellcheckDictionaryKey(WORKFLOW_NAME, testString, fieldName).getQuery();
    assertEquals(
      fieldName,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SpellingParams.SPELLCHECK_PREFIX + WORKFLOW_NAME + "." + testString));
  }

  public void test_setTerms() {
    final Record record = new SolrQueryBuilder().setTerms(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(TermsParams.TERMS).booleanValue());
  }

  public void test_addTermsField() {
    final Record record = new SolrQueryBuilder().addTermsField(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_FIELD));
  }

  public void test_setTermsLower() {
    final Record record = new SolrQueryBuilder().setTermsLower(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_LOWER));
  }

  public void test_setTermsLowerInclusive() {
    final Record record = new SolrQueryBuilder().setTermsLowerInclusive(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(TermsParams.TERMS_LOWER_INCLUSIVE).booleanValue());
  }

  public void test_setTermsMincount() {
    final Record record = new SolrQueryBuilder().setTermsMincount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(TermsParams.TERMS_MINCOUNT).intValue());
  }

  public void test_setTermsMaxcount() {
    final Record record = new SolrQueryBuilder().setTermsMaxcount(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(TermsParams.TERMS_MAXCOUNT).intValue());
  }

  public void test_setTermsPrefix() {
    final Record record = new SolrQueryBuilder().setTermsPrefix(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_PREFIX_STR));
  }

  public void test_setTermsRegex() {
    final Record record = new SolrQueryBuilder().setTermsRegex(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_REGEXP_STR));
  }

  public void test_addTermsRegexFlag() {
    final TermsRegexFlag trf = TermsRegexFlag.DOTALL;
    final Record record = new SolrQueryBuilder().addTermsRegexFlag(trf).getQuery();
    assertEquals(
      trf.toString(),
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_REGEXP_FLAG));
  }

  public void test_addTermsRegexFlagFail() {
    try {
      new SolrQueryBuilder().addTermsRegexFlag(null).getQuery();
    } catch (final Exception e) {
      if (e instanceof IllegalArgumentException) {
        return;
      }
    }
    fail();
  }

  public void test_setTermsLimit() {
    final Record record = new SolrQueryBuilder().setTermsLimit(testInt).getQuery();
    assertEquals(testInt, record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE)
      .getMap(QueryParams.QUERY).getLongValue(TermsParams.TERMS_LIMIT).intValue());
  }

  public void test_setTermsUpper() {
    final Record record = new SolrQueryBuilder().setTermsUpper(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_UPPER));
  }

  public void test_setTermsUpperInclusive() {
    final Record record = new SolrQueryBuilder().setTermsUpperInclusive(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(TermsParams.TERMS_UPPER_INCLUSIVE).booleanValue());
  }

  public void test_setTermsRaw() {
    final Record record = new SolrQueryBuilder().setTermsRaw(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(TermsParams.TERMS_RAW).booleanValue());
  }

  public void test_setTermsSort() {
    final TermsSort trf = TermsSort.INDEX;
    final Record record = new SolrQueryBuilder().setTermsSort(trf).getQuery();
    assertEquals(
      trf.toString(),
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(TermsParams.TERMS_SORT));
  }

  public void test_setShards() {
    final String[] similarityFields = new String[3];
    similarityFields[0] = testString;
    similarityFields[1] = fieldName;
    similarityFields[2] = WORKFLOW_NAME;
    final Record record = new SolrQueryBuilder().setShards(similarityFields).getQuery();
    assertEquals(
      testString + "," + fieldName + "," + WORKFLOW_NAME,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(ShardParams.SHARDS));
  }

  public void test_setShardsRequestHandler() {
    final Record record = new SolrQueryBuilder().setShardsRequestHandler(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(ShardParams.SHARDS_QT));
  }

  public void test_setShardsInfo() {
    final Record record = new SolrQueryBuilder().setShardsInfo(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(ShardParams.SHARDS_INFO).booleanValue());
  }

  public void test_setShardsTolerant() {
    final Record record = new SolrQueryBuilder().setShardsTolerant(testBoolean).getQuery();
    assertEquals(
      testBoolean,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getBooleanValue(ShardParams.SHARDS_TOLERANT).booleanValue());
  }

  public void test_setCursorMarkStart() {
    final Record record = new SolrQueryBuilder().setCursorMarkStart().getQuery();
    assertEquals(
      CursorMarkParams.CURSOR_MARK_START,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CursorMarkParams.CURSOR_MARK_PARAM));
  }

  public void test_setCursorMark() {
    final Record record = new SolrQueryBuilder().setCursorMark(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(CursorMarkParams.CURSOR_MARK_PARAM));
  }

  public void test_setCollection() {
    final Record record = new SolrQueryBuilder().setCollection(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SolrConstants.COLLECTION));
  }

  public void test_setRoute() {
    final Record record = new SolrQueryBuilder().setRoute(testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(SolrConstants.ROUTE));
  }

  public void test_setFinadAll() {
    final SolrQueryBuilder solrQueryBuilder = new SolrQueryBuilder();
    solrQueryBuilder.setQueryFindAll();
    final Record record = solrQueryBuilder.getQuery();
    assertEquals("*:*", record.getMetadata().getStringValue(QueryConstants.QUERY));
  }

  public void test_setNative() {
    final Record record = new SolrQueryBuilder().setNative(fieldName, testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(fieldName));
  }

  public void test_setNative2() {
    final Record record =
      new SolrQueryBuilder().setNative(fieldName, testString).setNative(fieldName, null).getQuery();
    assertNotSame(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue(fieldName));

  }

  public void test_setNative3() {
    final Record record = new SolrQueryBuilder().setNative(WORKFLOW_NAME, testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getStringValue("f." + fieldName + "." + WORKFLOW_NAME));
  }

  public void test_addNative() {
    final Record record = new SolrQueryBuilder().addNative(WORKFLOW_NAME, testString).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getSeq(WORKFLOW_NAME).get(0).asValue().asString());
  }

  public void test_addNative2() {
    final Record record = new SolrQueryBuilder().addNative(WORKFLOW_NAME, testString, fieldName).getQuery();
    assertEquals(
      testString,
      record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
        .getSeq("f." + fieldName + "." + WORKFLOW_NAME).get(0).asValue().asString());
  }

  public void test_addNative3() {
    final Record record =
      new SolrQueryBuilder().addNative(WORKFLOW_NAME, null).addNative(fieldName, "test").getQuery();
    assertNull(record.getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE).getMap(QueryParams.QUERY)
      .getSeq(WORKFLOW_NAME));
  }
}
