/**
 *
 */
package org.eclipse.smila.solr.update;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import junit.framework.TestCase;

import org.apache.solr.common.SolrInputDocument;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.solr.params.UpdateParams;
import org.eclipse.smila.solr.update.SolrDocumentConverter;

/**
 * @author pwissel
 *
 */
public class SolrDocumentConverter_Test extends TestCase {

  private static final String CONTENT = "Content";
  private static final String _RECORDID = "_recordid";
  private final DataFactory _factory = DataFactory.DEFAULT;

  public void test_toSolrDocumentMapping() throws IOException {
    final Record record = _factory.createRecord("mapping");

    final AnyMap metadata = record.getMetadata();
    final AnyMap mapping = _factory.createAnyMap();
    {
      metadata.put("title", "mapping-title");
      metadata.put("description", "mapping-description");
      metadata.put("other", "other-contetn");
      metadata.put("toAttachment", "attachment-value");
      final String content = "All your base are belong to us!";
      record.setAttachment(CONTENT, content.getBytes());
    }

    mapping.put("title", "Title");
    mapping.put("description", "");
    final AnySeq seq = mapping.getSeq("other", true);
    seq.add("title2");
    seq.add("title3");
    final AnyMap map = mapping.getMap(CONTENT, true);
    map.put(SolrDocumentConverter.FIELD_NAME, "Attachment");
    map.put(SolrDocumentConverter.FIELD_BOOST, 23.0f);
    map.put(SolrDocumentConverter.TYPE, SolrDocumentConverter.Type.ATTACHMENT.name());

    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record, mapping);
    assertEquals("mapping", sid.get(_RECORDID).getValue());
    assertEquals("mapping-title", sid.get("Title").getValue());
    assertEquals("mapping-description", sid.getField("description").getValue());
    assertEquals("other-contetn", sid.getFieldValue("title2"));
    assertEquals("other-contetn", sid.getFieldValue("title3"));
    assertEquals("All your base are belong to us!", sid.getFieldValue("Attachment"));
    assertNull(sid.getFieldValue("other"));
    assertEquals(23.0f, sid.getField("Attachment").getBoost());
  }

  private Record getSimpleRecord(final String id, final String title) {
    final Record record = _factory.createRecord(id);
    record.getMetadata().put("title", title);
    return record;
  }

  public void test_toSolrDocumentSimple() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("simple", "simple-title");
    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record);
    assertEquals("simple", sid.getFieldValue(_RECORDID));
    assertEquals("simple-title", sid.getFieldValue("title"));
  }

  public void test_toSolrDocumentWithChildren() throws IOException {
    final Record record = getSimpleRecord("parent", "Parent");
    record.setAttachment(CONTENT, "parentsContent".getBytes());
    final AnyMap mapping = _factory.createAnyMap();
    mapping.put("title", "Title");
    mapping.put("description", "");
    final AnySeq seq = mapping.getSeq("other", true);
    seq.add("title2");
    seq.add("title3");
    final AnyMap map = mapping.getMap(CONTENT, true);
    map.put(SolrDocumentConverter.FIELD_NAME, "Attachment");
    map.put(SolrDocumentConverter.FIELD_BOOST, 23.0f);
    map.put(SolrDocumentConverter.TYPE, SolrDocumentConverter.Type.ATTACHMENT.name());
    final List<Record> childs = new ArrayList<>();
    final Record child1 = getSimpleRecord("child1", "Children1");
    child1.getMetadata().put("desc", "text");
    childs.add(child1);
    final Record child2 = getSimpleRecord("child2", "Children2");
    childs.add(child2);
    final Record child3 = getSimpleRecord("child3", "Children3");
    childs.add(child3);
    child1.setAttachment(CONTENT, "child1sContent".getBytes());
    child2.setAttachment(CONTENT, "child2sContent".getBytes());
    child3.setAttachment(CONTENT, "child3sContent".getBytes());
    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record, mapping, childs);

    assertEquals("parent", sid.getFieldValue(_RECORDID));
    final List<SolrInputDocument> sids = sid.getChildDocuments();
    assertEquals("child1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Children1", sids.get(0).getFieldValue("Title"));
    assertEquals("child2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Children2", sids.get(1).getFieldValue("Title"));
    assertEquals("child3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Children3", sids.get(2).getFieldValue("Title"));
  }

  public void test_toSolrDocumentWithMappingWithChildren() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("parent", "Parent");
    final List<Record> childs = new ArrayList<>();
    childs.add(getSimpleRecord("child1", "Children1"));
    childs.add(getSimpleRecord("child2", "Children2"));
    childs.add(getSimpleRecord("child3", "Children3"));
    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record, childs);

    assertEquals("parent", sid.getFieldValue(_RECORDID));
    final List<SolrInputDocument> sids = sid.getChildDocuments();
    assertEquals("child1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Children1", sids.get(0).getFieldValue("title"));
    assertEquals("child2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Children2", sids.get(1).getFieldValue("title"));
    assertEquals("child3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Children3", sids.get(2).getFieldValue("title"));
  }

  public void test_toSolrDocumentWithChildrenWithAttachment() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("parent", "Parent");
    final List<Record> childs = new ArrayList<>();
    childs.add(getSimpleRecord("child1", "Children1"));
    childs.add(getSimpleRecord("child2", "Children2"));
    childs.add(getSimpleRecord("child3", "Children3"));
    for (Record record2 : childs) {
      record2.setAttachment(CONTENT, "contentcontentcontent".getBytes());
    }
    SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record, false, childs);

    assertEquals("parent", sid.getFieldValue(_RECORDID));
    List<SolrInputDocument> sids = sid.getChildDocuments();
    assertEquals("child1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Children1", sids.get(0).getFieldValue("title"));
    assertEquals("child2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Children2", sids.get(1).getFieldValue("title"));
    assertEquals("child3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Children3", sids.get(2).getFieldValue("title"));

    for (SolrInputDocument solrInputDocument : sids) {
      assertNull(solrInputDocument.getFieldValue(CONTENT));
    }

    sid = new SolrDocumentConverter().toSolrDocument(record, true, childs);

    assertEquals("parent", sid.getFieldValue(_RECORDID));
    sids = sid.getChildDocuments();
    assertEquals("child1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Children1", sids.get(0).getFieldValue("title"));
    assertEquals("child2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Children2", sids.get(1).getFieldValue("title"));
    assertEquals("child3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Children3", sids.get(2).getFieldValue("title"));

    for (SolrInputDocument solrInputDocument : sids) {
      assertNotNull(solrInputDocument.getFieldValue(CONTENT));
    }
  }

  public void test_toSolrDocumentWithManyRecords() throws UnsupportedEncodingException {
    final List<Record> records = new ArrayList<>();
    records.add(getSimpleRecord("Number1", "Title-No1"));
    records.add(getSimpleRecord("Number2", "Title-No2"));
    records.add(getSimpleRecord("Number3", "Title-No3"));

    final List<SolrInputDocument> sids = new SolrDocumentConverter().toSolrDocuments(records);
    assertEquals("Number1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Title-No1", sids.get(0).getFieldValue("title"));
    assertEquals("Number2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Title-No2", sids.get(1).getFieldValue("title"));
    assertEquals("Number3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Title-No3", sids.get(2).getFieldValue("title"));
  }

  public void test_toSolrDocumentWithManyRecordsAttachment() throws UnsupportedEncodingException {
    final List<Record> records = new ArrayList<>();
    final String content = "Sie sagten, ich könne niemals einem Lama das Autofahren beibringen!";
    final Record number1 = getSimpleRecord("Number1", "Title-No1");
    number1.setAttachment(CONTENT, content.getBytes("utf-8"));
    records.add(number1);
    final Record number2 = getSimpleRecord("Number2", "Title-No2");
    number2.setAttachment(CONTENT, content.getBytes("utf-8"));
    records.add(number2);
    final Record number3 = getSimpleRecord("Number3", "Title-No3");
    number3.setAttachment(CONTENT, content.getBytes("utf-8"));
    records.add(number3);

    final List<SolrInputDocument> sids = new SolrDocumentConverter().toSolrDocuments(records, true);
    assertEquals("Number1", sids.get(0).getFieldValue(_RECORDID));
    assertEquals("Title-No1", sids.get(0).getFieldValue("title"));
    assertEquals("Number2", sids.get(1).getFieldValue(_RECORDID));
    assertEquals("Title-No2", sids.get(1).getFieldValue("title"));
    assertEquals("Number3", sids.get(2).getFieldValue(_RECORDID));
    assertEquals("Title-No3", sids.get(2).getFieldValue("title"));
    assertEquals(content, sids.get(0).getFieldValue(CONTENT));
    assertEquals(content, sids.get(1).getFieldValue(CONTENT));
    assertEquals(content, sids.get(2).getFieldValue(CONTENT));
  }

  public void test_toSolrDocumentAttachment() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("recordid", "Record-with-attachment");
    final String content =
      "If ever there is tomorrow when we're not together.. there is something you must always remember. you are braver than you believe, stronger than you seem, and smarter than you think. but the most important thing is, even if we're apart.. i'll always be with you.";
    record.setAttachment(CONTENT, content.getBytes());
    final SolrInputDocument sid_noContent = new SolrDocumentConverter().toSolrDocument(record, false);
    assertNull(sid_noContent.getFieldValue(CONTENT));
    final SolrInputDocument sid_Content = new SolrDocumentConverter().toSolrDocument(record, true);
    assertNotNull(sid_Content.getFieldValue(CONTENT));
    assertEquals(content, sid_Content.getFieldValue(CONTENT));
  }

  public void test_toSolrDocumentParam() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("param", "Parameters");
    final AnyMap metadata = record.getMetadata();
    final AnyMap solr = metadata.getMap("_solr", true);
    final AnyMap importing = solr.getMap(UpdateParams.UPDATE, true);
    importing.put(UpdateParams.DOCUMENT_BOOST, 23.0f);
    {
      final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record);
      assertEquals(23.0f, sid.getDocumentBoost());
    }
    importing.remove(UpdateParams.DOCUMENT_BOOST);
    {
      final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record);
      assertEquals(1.0f, sid.getDocumentBoost());
    }
  }

  public void test_toSolrDocumentValueTyes() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("Values", "Different Values");
    final AnyMap metadata = record.getMetadata();

    final AnySeq seq = metadata.getSeq("sequence", true);
    seq.add(23);
    seq.add("22");
    final AnyMap map = metadata.getMap("map", true);
    map.put("key", "value");
    map.put("nextKey", "nextValue");
    metadata.put("string", "string-value");
    final Value doubleVal = _factory.createDoubleValue(987162387512.817236417856);
    final Value booleanVal = _factory.createBooleanValue(true);
    final Value longVal = _factory.createLongValue(Long.MAX_VALUE);
    final Value dateVal = _factory.createDateValue(new Date(1234567890));
    final Value dateTimeVal = _factory.createDateTimeValue(new Date(1234567890));
    metadata.put("double", doubleVal);
    metadata.put("boolean", booleanVal);
    metadata.put("long", longVal);
    metadata.put("date", dateVal);
    metadata.put("dateTime", dateTimeVal);

    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record);
    assertEquals("Values", sid.getFieldValue(_RECORDID));
    assertTrue(sid.getField("sequence").getValues().contains(new Long(23)));
    assertTrue(sid.getField("sequence").getValues().contains("22"));
    assertNull(sid.getFieldValue("map"));
    assertEquals(new Double(987162387512.817236417856), sid.getFieldValue("double"));
    assertEquals(new Boolean(true), sid.getFieldValue("boolean"));
    assertEquals(new Long(Long.MAX_VALUE), sid.getFieldValue("long"));
    assertEquals(new Date(1234567890), sid.getFieldValue("date"));
    assertEquals(new Date(1234567890), sid.getFieldValue("dateTime"));

  }

  public void test_toSolrDocumentValueTyesWithMapping() throws UnsupportedEncodingException {
    final Record record = getSimpleRecord("Values", "Different Values");
    final AnyMap metadata = record.getMetadata();
    // MAP, SEQ, STRING, DOUBLE, BOOLEAN, LONG, DATE, DATETIME

    final AnyMap mapping = metadata.getMap("mapping", true);
    mapping.put("map", "");
    mapping.put("sequence", "");
    mapping.put("string", "");
    mapping.put("double", "");
    mapping.put("date", "Date");
    mapping.put("dateTime", "");
    mapping.put("long", "");
    mapping.put("boolean", "");

    final AnySeq seq = metadata.getSeq("sequence", true);
    seq.add(23);
    seq.add("22");
    final AnyMap map = metadata.getMap("map", true);
    map.put("key", "value");
    map.put("nextKey", "nextValue");
    metadata.put("string", "string-value");
    final Value doubleVal = _factory.createDoubleValue(987162387512.817236417856);
    final Value booleanVal = _factory.createBooleanValue(true);
    final Value longVal = _factory.createLongValue(Long.MAX_VALUE);
    final Value dateVal = _factory.createDateValue(new Date(1234567890));
    final Value dateTimeVal = _factory.createDateTimeValue(new Date(1234567890));
    metadata.put("double", doubleVal);
    metadata.put("boolean", booleanVal);
    metadata.put("long", longVal);
    metadata.put("date", dateVal);
    metadata.put("dateTime", dateTimeVal);

    final SolrInputDocument sid = new SolrDocumentConverter().toSolrDocument(record, mapping);
    assertEquals("Values", sid.getFieldValue(_RECORDID));
    assertTrue(sid.getField("sequence").getValues().contains(new Long(23)));
    assertTrue(sid.getField("sequence").getValues().contains("22"));
    assertNull(sid.getFieldValue("map"));
    assertEquals(new Double(987162387512.817236417856), sid.getFieldValue("double"));
    assertEquals(new Boolean(true), sid.getFieldValue("boolean"));
    assertEquals(new Long(Long.MAX_VALUE), sid.getFieldValue("long"));
    assertEquals(new Date(1234567890), sid.getFieldValue("Date"));
    assertEquals(new Date(1234567890), sid.getFieldValue("dateTime"));

  }

}
