/**
 *
 */
package org.eclipse.smila.solr.httphandler;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.json.JsonHttpUtils;
import org.eclipse.smila.http.server.util.ARequestHandler;
import org.eclipse.smila.solr.SolrServerService;
import org.eclipse.smila.solr.server.ServerUtil;

/**
 * @author pwissel
 *
 */
public class ServerHandler extends ARequestHandler {

  private final JsonHttpUtils _jsonHttpUtils = new JsonHttpUtils();

  private SolrServerService _serverService;

  /*
   * (non-Javadoc)
   *
   * @see org.eclipse.smila.http.server.HttpHandler#handle(org.eclipse.smila.http.server.HttpExchange)
   */
  @Override
  public void handle(HttpExchange exchange) throws IOException {
    // read from exchange
    final String requestUri = exchange.getRequestURI();
    final Map<String, String[]> params = exchange.getParameterMap();
    final String index = getIndex(requestUri);
    final String operation = getOperation(requestUri);
    // get solr server
    SolrServer server = null;
    AnyMap result = null;
    try {
      try {
        server = _serverService.getServer(index);
      } catch (final SolrServerException exception) {
        throw new IllegalArgumentException("can not get solr server for index: " + index, exception);
      }
      // execute operation
      final boolean waitFlush = BooleanUtils.toBoolean(getParameter(params, "waitFlush"));
      final boolean waitSearcher = BooleanUtils.toBoolean(getParameter(params, "waitSearcher"));
      switch (operation) {
        case "commit":
          final boolean softCommit = BooleanUtils.toBoolean(getParameter(params, "softCommit"));
          result = ServerUtil.commit(server, waitFlush, waitSearcher, softCommit);
          break;
        case "optimize":
          final int maxSegments = NumberUtils.toInt(getParameter(params, "maxSegments"), 1);
          result = ServerUtil.optimize(server, waitFlush, waitSearcher, maxSegments);
          break;
        default:
          throw new IllegalArgumentException("operation unknown: " + operation);
      }
    } catch (Exception exception) {
      _jsonHttpUtils.writeErrorDetails(exchange, exception);
      return;
    }
    _jsonHttpUtils.writeResultObject(exchange.getResponseStream(), result);
    exchange.getResponseStream().flush();
  }

  private String getIndex(final String requestUri) {
    try {
      return getDynamicUriParts(requestUri).get(0);
    } catch (final Exception exception) {
      throw new IllegalArgumentException("request uri does not contain index");
    }
  }

  private String getOperation(final String requestUri) {
    try {
      return getDynamicUriParts(requestUri).get(1);
    } catch (final Exception exception) {
      throw new IllegalArgumentException("request uri does not contain operation");
    }
  }

  private String getParameter(final Map<String, String[]> params, final String name) {
    final String[] param = params.get(name);
    if (param != null && param.length < 0) {
      return param[0];
    }
    return null;
  }

  protected void bindSolrServerService(final SolrServerService serverService) {
    if (_log.isDebugEnabled()) {
      _log.debug("Bind SolrServerService...");
    }
    _serverService = serverService;
  }

  protected void unbindSolrServerService(final SolrServerService serverService) {
    if (_serverService == serverService) {
      if (_log.isDebugEnabled()) {
        _log.debug("Unbind SolrServerService...");
      }
      _serverService = null;
    }
  }

}
