/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.test;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.AnyConvertException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;

import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;

/**
 * 
 */
public class TestResultDescription extends TestCase {

  /**
   * Tests the conversion of ResultDescription from and to an Any object.
   * 
   * @throws Exception
   *           exception converting Any.
   */
  public void testResultDescriptionFromAndToAny() throws Exception {
    final String errorCode = "Error Code";
    final String errorMessage = "Error Message";
    final Integer integerNumber = Integer.valueOf(1000);
    final Double doubleNumber = Double.valueOf(100);
    final AnyMap countersAny = DataFactory.DEFAULT.createAnyMap();
    final String integerKey = "integer";
    countersAny.put(integerKey, integerNumber);
    final String doubleKey = "double";
    countersAny.put(doubleKey, doubleNumber);
    final AnyMap resultDescriptionAny = DataFactory.DEFAULT.createAnyMap();
    resultDescriptionAny.put(ResultDescription.KEY_STATUS, TaskCompletionStatus.SUCCESSFUL.name());
    resultDescriptionAny.put(ResultDescription.KEY_ERROR_CODE, errorCode);
    resultDescriptionAny.put(ResultDescription.KEY_ERROR_MESSAGE, errorMessage);
    resultDescriptionAny.put(ResultDescription.KEY_COUNTERS, countersAny);

    final ResultDescription resDes = ResultDescription.fromAny(resultDescriptionAny);
    final AnyMap reconvertedResDes = resDes.toAny();

    assertEquals("Wrong status", resultDescriptionAny.get(ResultDescription.KEY_STATUS),
      reconvertedResDes.get(ResultDescription.KEY_STATUS));
    assertEquals("Wrong error code", resultDescriptionAny.get(ResultDescription.KEY_ERROR_CODE),
      reconvertedResDes.get(ResultDescription.KEY_ERROR_CODE));
    assertEquals("Wrong error message", resultDescriptionAny.get(ResultDescription.KEY_ERROR_MESSAGE),
      reconvertedResDes.get(ResultDescription.KEY_ERROR_MESSAGE));
    assertEquals("Wrong integer counter value",
      resultDescriptionAny.getMap(ResultDescription.KEY_COUNTERS).get(integerKey),
      reconvertedResDes.getMap(ResultDescription.KEY_COUNTERS).get(integerKey));
    assertEquals("Wrong double counter value",
      resultDescriptionAny.getMap(ResultDescription.KEY_COUNTERS).get(doubleKey),
      reconvertedResDes.getMap(ResultDescription.KEY_COUNTERS).get(doubleKey));
  }

  /**
   * Tests the conversion of ResultDescription with only status set from and to an Any object.
   * 
   * @throws Exception
   *           exception converting Any.
   */
  public void testResultDescriptionWithOnlyStatusSetFromAndToAny() throws Exception {
    final AnyMap resultDescriptionAny = DataFactory.DEFAULT.createAnyMap();
    resultDescriptionAny.put(ResultDescription.KEY_STATUS, TaskCompletionStatus.SUCCESSFUL.name());

    final ResultDescription resDes = ResultDescription.fromAny(resultDescriptionAny);
    final AnyMap reconvertedResDes = resDes.toAny();

    assertEquals("Wrong status", resultDescriptionAny.get(ResultDescription.KEY_STATUS),
      reconvertedResDes.get(ResultDescription.KEY_STATUS));
    assertNull("Error code unexpected", resultDescriptionAny.get(ResultDescription.KEY_ERROR_CODE));
    assertNull("Error message unexpected", resultDescriptionAny.get(ResultDescription.KEY_ERROR_MESSAGE));
    assertNull("Counters unexpected", resultDescriptionAny.get(ResultDescription.KEY_COUNTERS));
  }

  /**
   * Tests the conversion of ResultDescription with no status set from an Any object.
   * 
   * @throws Exception
   *           exception converting Any.
   */
  public void testResultDescriptionWithNoStatusSetFromAny() throws Exception {
    final AnyMap resultDescriptionAny = DataFactory.DEFAULT.createAnyMap();
    resultDescriptionAny.put(ResultDescription.KEY_STATUS + "blub", TaskCompletionStatus.SUCCESSFUL.name());

    try {
      ResultDescription.fromAny(resultDescriptionAny);
      fail("must fail when there's no status.");
    } catch (final Exception ex) {
      assertTrue("wrong exception", ex instanceof AnyConvertException);
    }
  }

  /**
   * Tests the conversion of ResultDescription with invalid counter value from an Any object.
   * 
   * @throws Exception
   *           exception converting Any.
   */
  public void testResultDescriptionWithInvalidCounterFromAny() throws Exception {
    final AnyMap resultDescriptionAny = DataFactory.DEFAULT.createAnyMap();
    resultDescriptionAny.put(ResultDescription.KEY_STATUS, TaskCompletionStatus.SUCCESSFUL.name());
    final AnyMap counterAny = DataFactory.DEFAULT.createAnyMap();
    counterAny.put("x", "y");
    resultDescriptionAny.put(ResultDescription.KEY_COUNTERS, counterAny);

    try {
      ResultDescription.fromAny(resultDescriptionAny);
      fail("must fail when there's an invalid counter value.");
    } catch (final Exception ex) {
      assertTrue("wrong exception", ex instanceof AnyConvertException);
    }
  }
}
