/*******************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * <pre>
 * Contributors: Jürgen Schumacher (Empolis Information Management GmbH) - implementation
 * </pre>
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test for TaskManager performance. Increate timeToLive in clusterconfig.json to make this work.
 * 
 */
public class TestTaskManagerFairness extends DeclarativeServiceTestCase {

  /** name of final worker. */
  private static final String TEST_WORKER = "TestTaskManagerFairness";

  /** reference to task manager service. */
  private TaskManager _taskManager;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _taskManager = getService(TaskManager.class);
  }

  public void testPutAndGetTasksNonQualifiedFair() throws Exception {

    final List<Task> retrievedTasks = new ArrayList<Task>();

    putAndGetTasksNonQualified(retrievedTasks);

    // check first 10
    Map<String, String> taskProps = retrievedTasks.get(0).getProperties();
    String lastName = taskProps.get(Task.PROPERTY_JOB_NAME);
    for (int i = 1; i < 10; i++) {
      taskProps = retrievedTasks.get(i).getProperties();
      assertTrue(taskProps.containsKey(Task.PROPERTY_JOB_NAME));
      final String jobName = taskProps.get(Task.PROPERTY_JOB_NAME);
      assertFalse(jobName.equals(lastName));
      lastName = jobName;
    }

    // expect "round robbin fairness"

    int countA = 0;
    int countB = 0;

    for (int i = 0; i < 100; i++) {
      taskProps = retrievedTasks.get(i).getProperties();
      assertTrue(taskProps.containsKey("jobName"));
      final String jobName = taskProps.get("jobName");
      if (jobName.equals("job0")) {
        ++countA;
      } else {
        ++countB;
      }
    }
    // within the first 100 tasks, we find all B tasks and 50 A tasks
    assertEquals(5, countB);
    assertEquals(95, countA);
  }

  /*
   * create one large (A) and one small (B) job (100 vs 5) Push all A tasks and then all B tasks.
   */
  private void putAndGetTasksNonQualified(final List<Task> retrievedTasks) throws Exception {
    final List<Integer> jobSizes = new ArrayList<Integer>();
    jobSizes.add(100);
    jobSizes.add(5);
    final int noOfJobs = 2;
    final Collection<Task> tasks = new ArrayList<Task>();

    for (int i = 0; i < noOfJobs; i++) {
      for (int j = 0; j < jobSizes.get(i).intValue(); j++) {
        tasks.add(createTask("job" + i));
      }
    }

    _taskManager.addTasks(tasks);

    for (int i = 0; i < tasks.size(); i++) {
      retrievedTasks.add(_taskManager.getTask(TEST_WORKER, null));
    }

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }

    assertEquals(105, retrievedTasks.size());
  }

  // debug

  private void printTasks(final List<Task> retrievedTasks) {
    for (int i = 0; i < retrievedTasks.size(); ++i) {
      final Task _task = retrievedTasks.get(i);
      final Map<String, String> taskProps = _task.getProperties();
      System.out.println("Task: " + taskProps.get(Task.PROPERTY_JOB_NAME) + " / " + _task.getQualifier());
    }
  }

  private void addTask(final String jobName, final String qualifier, final List<Task> tasks) {
    final Task task = createTask(jobName);
    task.setQualifier(qualifier);
    tasks.add(task);
  }

  private boolean checkTask(final Task task, final String expectedJobName, final String expectedQualifier) {
    final Map<String, String> taskProps = task.getProperties();
    if (taskProps.get(Task.PROPERTY_JOB_NAME).equals(expectedJobName)
      && task.getQualifier().equals(expectedQualifier)) {
      return true;
    }
    return false;
  }

  /*
   * test for fairness when qualifiers for the designated "next job" don't match and the TM has to provide tasks from
   * the following jobs
   */
  public void testPutAndGetTasksQualifiedFairUnmatchedQualifier() throws Exception {

    final List<Task> tasks = new ArrayList<Task>();

    /*
     * job0 * 3 tasks * first added * qualifiers 2,3 job1 * 2 tasks * added last * qualifiers 1
     */

    addTask("job0", "2", tasks);
    addTask("job0", "3", tasks);
    addTask("job0", "2", tasks);

    addTask("job1", "1", tasks);
    addTask("job1", "1", tasks);
    _taskManager.addTasks(tasks);

    final List<String> qualifiers1 = new ArrayList<>(1);
    qualifiers1.add("1");

    final List<String> qualifiers2_3 = new ArrayList<>(2);
    qualifiers2_3.add("2");
    qualifiers2_3.add("3");

    // debug
    final List<String> qualifiersFull = new ArrayList<>(3);
    qualifiersFull.add("1");
    qualifiersFull.add("2");
    qualifiersFull.add("3");

    /*
     * The order for a fullQualifier-Request would be Task: job1 / 1 Task: job1 / 1 Task: job0 / 2 Task: job0 / 2 Task:
     * job0 / 3
     */

    {
      final Task rTask = _taskManager.getTask(TEST_WORKER, null, qualifiers2_3);
      assertTrue(checkTask(rTask, "job0", "2"));
    }
    {
      final Task rTask = _taskManager.getTask(TEST_WORKER, null, qualifiers2_3);
      assertTrue(checkTask(rTask, "job0", "2"));
    }
    {
      final Task rTask = _taskManager.getTask(TEST_WORKER, null, qualifiers1);
      assertTrue(checkTask(rTask, "job1", "1"));
    }
    {
      final Task rTask = _taskManager.getTask(TEST_WORKER, null, qualifiers2_3);
      assertTrue(checkTask(rTask, "job0", "3"));
    }
    // making it empty
    {
      final Task rTask = _taskManager.getTask(TEST_WORKER, null, qualifiers1);
      assertTrue(checkTask(rTask, "job1", "1"));
    }

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }
  }

  /*
   * Test for "fair" task handling in the case of three equally sized jobs
   * 
   * Note: It is not guaranteed, that the TM will provide the qualifiers in order
   */
  public void testPutAndGetTasksQualifiedFair() throws Exception {

    final List<Task> retrievedTasks = new ArrayList<Task>();

    final List<Integer> jobSizes = new ArrayList<Integer>();
    jobSizes.add(20);
    jobSizes.add(20);
    jobSizes.add(20);
    final int noOfQualifiers = 5;

    putAndGetTasksQualified(jobSizes, noOfQualifiers, retrievedTasks);

    assertEquals(60, retrievedTasks.size());

    int totalCount = 0;

    // start
    Task _task = retrievedTasks.get(totalCount);
    Map<String, String> taskProps = _task.getProperties();
    String secondToLast = taskProps.get(Task.PROPERTY_JOB_NAME);
    ++totalCount;
    _task = retrievedTasks.get(totalCount);
    taskProps = _task.getProperties();
    String last = taskProps.get(Task.PROPERTY_JOB_NAME);
    ++totalCount;
    assertFalse(last.equals(secondToLast));

    while (true) {
      if (totalCount == 60) {
        break;
      }
      _task = retrievedTasks.get(totalCount);
      taskProps = _task.getProperties();
      final String thisName = taskProps.get(Task.PROPERTY_JOB_NAME);
      assertFalse(thisName.equals(secondToLast));
      assertFalse(thisName.equals(last));
      ++totalCount;
      // shift
      secondToLast = last;
      last = thisName;
    }
  }

  /*
   * create three jobs (each 20 tasks) Push all A tasks, then all B tasks, then all C tasks.
   * 
   * Each task gets one of 5 qualifiers.
   */
  private void putAndGetTasksQualified(final List<Integer> jobSizes, final int noOfQualifiers,
    final List<Task> retrievedTasks) throws Exception {
    final int noOfJobs = jobSizes.size();

    final List<String> qualifiers = new ArrayList<>(noOfQualifiers);
    for (int i = 0; i < noOfQualifiers; i++) {
      qualifiers.add(Integer.toString(i));
    }

    final Collection<Task> tasks = new ArrayList<Task>();

    for (int i = 0; i < noOfJobs; i++) {
      for (int j = 0; j < jobSizes.get(i).intValue(); j++) {
        final Task task = createTask("job" + i);
        task.setQualifier(qualifiers.get(j % noOfQualifiers));
        tasks.add(task);
      }
    }

    System.out.println("Adding " + tasks.size() + " tasks for " + noOfJobs + " jobs with " + noOfQualifiers
      + " qualifiers...");
    final long startAdd = System.nanoTime();
    _taskManager.addTasks(tasks);
    final double timeAddMs = (System.nanoTime() - startAdd) / 1e6;
    System.out.println("Times: complete " + timeAddMs + " ms, per task " + timeAddMs / tasks.size() + " ms");

    System.out.println("Getting " + tasks.size() + " tasks...");
    final long startGet = System.nanoTime();

    for (int i = 0; i < tasks.size(); i++) {
      retrievedTasks.add(_taskManager.getTask(TEST_WORKER, null, qualifiers));
      assertNotNull("too few tasks available", retrievedTasks.listIterator(retrievedTasks.size() - 1));
    }

    final double timeGetMs = (System.nanoTime() - startGet) / 1e6;
    System.out.println("Times: complete " + timeGetMs + " ms, per task " + timeGetMs / tasks.size() + " ms");

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }
  }

  private Task createTask(final String jobName) {
    return createTask(jobName, TEST_WORKER);
  }

  private Task createTask(final String jobName, final String workerName) {
    final Task task = new Task(UUID.randomUUID().toString(), workerName);
    task.getProperties().put(Task.PROPERTY_JOB_NAME, jobName);
    task.getProperties().put(Task.PROPERTY_JOB_RUN_ID, "1");
    task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, "1");
    return task;
  }

  /**
   * Test for task counters for non-qualified tasks.
   * 
   * @throws Exception
   *           on error
   */
  public void testTaskCountersForNonQualifiedTasks() throws Exception {
    doTaskCounters(false);
  }

  /**
   * Test for task counters for qualified tasks.
   * 
   * @throws Exception
   *           on error
   */
  public void testTaskCountersForQualifiedTasks() throws Exception {
    doTaskCounters(true);
  }

  private void doTaskCounters(final boolean qualified) throws Exception {
    // one worker, one job
    doTaskCounters(1, 1, 3, qualified);

    // one worker, two jobs
    doTaskCounters(1, 2, 3, qualified);

    // two worker, one job
    doTaskCounters(2, 1, 3, qualified);

    // two worker, two jobs
    doTaskCounters(2, 2, 3, qualified);
  }

  private void doTaskCounters(final int noOfWorkers, final int noOfJobs, final int noOfJobTasksPerWorker,
    final boolean qualified) throws Exception {
    final int noOfTasks = noOfJobs * noOfWorkers * noOfJobTasksPerWorker;
    prepareTasks(noOfWorkers, noOfJobs, noOfJobTasksPerWorker, qualified);
    consumeTasksAndAssertCounters(noOfWorkers, noOfTasks, qualified);
    assertEmptyTaskCounters(noOfWorkers, noOfJobs);
  }

  private void prepareTasks(final int noOfWorkers, final int noOfJobs, final int noOfJobTasksPerWorker,
    final boolean qualified) throws Exception {
    if (qualified) {
      prepareQualifiedTasks(noOfWorkers, noOfJobs, noOfJobTasksPerWorker);
    } else {
      prepareNonQualifiedTasks(noOfWorkers, noOfJobs, noOfJobTasksPerWorker);
    }
  }

  private void prepareNonQualifiedTasks(final int noOfWorkers, final int noOfJobs, final int noOfJobTasksPerWorker)
    throws Exception {
    final List<String> workerNames = getWorkerNames(noOfWorkers);
    final List<String> jobNames = getJobNames(noOfJobs);
    final List<Task> tasks = new ArrayList<Task>();
    for (int w = 0; w < workerNames.size(); w++) {
      for (int j = 0; j < jobNames.size(); j++) {
        for (int t = 0; t < noOfJobTasksPerWorker; t++) {
          tasks.add(createTask(jobNames.get(j), workerNames.get(w)));
        }
      }
    }
    _taskManager.addTasks(tasks);
  }

  private void prepareQualifiedTasks(final int noOfWorkers, final int noOfJobs, final int noOfJobTasksPerWorker)
    throws Exception {
    final List<String> qualifiers = getQualifiers();
    final List<String> workerNames = getWorkerNames(noOfWorkers);
    final List<String> jobNames = getJobNames(noOfJobs);
    final List<Task> tasks = new ArrayList<Task>();
    for (int w = 0; w < workerNames.size(); w++) {
      for (int j = 0; j < jobNames.size(); j++) {
        for (int t = 0; t < noOfJobTasksPerWorker; t++) {
          final Task task = createTask(jobNames.get(j), workerNames.get(w));
          task.setQualifier(qualifiers.get(t % qualifiers.size()));
          tasks.add(task);
        }
      }
    }
    _taskManager.addTasks(tasks);
  }

  private void consumeTasksAndAssertCounters(final int noOfWorkers, final int noOfTasks, final boolean qualified)
    throws Exception {

    final List<String> qualifiers = qualified ? getQualifiers() : null;

    final List<String> workerNames = getWorkerNames(noOfWorkers);

    for (int i = 0; i < noOfTasks; i++) {
      final String workerName = workerNames.get(i % workerNames.size());

      final Map<String, TaskCounter> oldTaskCounters = _taskManager.getTaskCounters();

      final Task task =
        qualified ? _taskManager.getTask(workerName, null, qualifiers) : _taskManager.getTask(workerName, null);

      final String jobName = task.getProperties().get(Task.PROPERTY_JOB_NAME);

      final Map<String, TaskCounter> newTaskCounters = _taskManager.getTaskCounters();

      final int oldTasksTodoForWorker = getTasksTodoForWorker(oldTaskCounters, workerName);
      final int oldTasksTodoForWorkerAndJob = getTasksTodoForWorkerAndJob(oldTaskCounters, workerName, jobName);

      final int newTasksTodoForWorker = getTasksTodoForWorker(newTaskCounters, workerName);
      final int newTasksTodoForWorkerAndJob = getTasksTodoForWorkerAndJob(newTaskCounters, workerName, jobName);

      assertEquals(newTasksTodoForWorker, oldTasksTodoForWorker - 1);
      assertEquals(newTasksTodoForWorkerAndJob, oldTasksTodoForWorkerAndJob - 1);

      assertEquals(0, getTasksInProgressForWorker(oldTaskCounters, workerName));
      assertEquals(1, getTasksInProgressForWorker(newTaskCounters, workerName));

      _taskManager.finishTask(task.getWorkerName(), task.getTaskId(), new ResultDescription(
        TaskCompletionStatus.SUCCESSFUL, null, null, null));
    }
  }

  private int getTasksTodoForWorker(final Map<String, TaskCounter> counters, final String workerName)
    throws Exception {
    assertTrue(counters.containsKey(workerName));
    return counters.get(workerName).getTasksTodo();
  }

  private int getTasksTodoForWorkerAndJob(final Map<String, TaskCounter> counters, final String workerName,
    final String jobName) throws Exception {
    assertTrue(counters.containsKey(workerName));
    final TaskCounter counter = counters.get(workerName);
    assertTrue(counter.getTasksTodoPerJobs().containsKey(jobName));
    return counter.getTasksTodoPerJobs().get(jobName).intValue();
  }

  private int getTasksInProgressForWorker(final Map<String, TaskCounter> counters, final String workerName)
    throws Exception {
    assertTrue(counters.containsKey(workerName));
    return counters.get(workerName).getTasksInProgress();
  }

  private void assertTasksTodoForWorker(final Map<String, TaskCounter> counters, final String workerName,
    final int expectedTasksTodo) throws Exception {
    assertEquals(expectedTasksTodo, getTasksTodoForWorker(counters, workerName));
  }

  private void assertTasksTodoForWorkerAndJob(final Map<String, TaskCounter> counters, final String workerName,
    final String jobName, final int expectedTasksTodo) {
    assertTrue(counters.containsKey(workerName));
    final TaskCounter counter = counters.get(workerName);
    assertTrue(counter.getTasksTodoPerJobs().containsKey(jobName));
    assertEquals(expectedTasksTodo, counter.getTasksTodoPerJobs().get(jobName).intValue());
  }

  private void assertTasksInProgressForWorker(final Map<String, TaskCounter> counters, final String workerName,
    final int expectedTasksInProgress) throws Exception {
    assertEquals(expectedTasksInProgress, getTasksInProgressForWorker(counters, workerName));
  }

  private void assertEmptyTaskCounters(final int noOfWorkers, final int noOfJobs) throws Exception {
    final List<String> workerNames = getWorkerNames(noOfWorkers);
    final List<String> jobNames = getJobNames(noOfJobs);
    final Map<String, TaskCounter> counters = _taskManager.getTaskCounters();
    for (int w = 0; w < workerNames.size(); w++) {
      final String workerName = workerNames.get(w);
      assertTasksTodoForWorker(counters, workerName, 0);
      assertTasksInProgressForWorker(counters, workerName, 0);
      for (int j = 0; j < jobNames.size(); ++j) {
        assertTasksTodoForWorkerAndJob(counters, workerName, jobNames.get(j), 0);
      }
    }
  }

  private List<String> getWorkerNames(final int noOfWorkers) {
    final List<String> workerNames = new ArrayList<>(noOfWorkers);
    for (int i = 0; i < noOfWorkers; i++) {
      workerNames.add("worker" + i);
    }
    return workerNames;
  }

  private List<String> getJobNames(final int noOfJobs) {
    final List<String> jobNames = new ArrayList<>(noOfJobs);
    for (int i = 0; i < noOfJobs; ++i) {
      jobNames.add("job" + i);
    }
    return jobNames;
  }

  private List<String> getQualifiers() {
    final int noOfQualifiers = 3;
    final List<String> qualifiers = new ArrayList<>(noOfQualifiers);
    for (int i = 0; i < noOfQualifiers; i++) {
      qualifiers.add(Integer.toString(i));
    }
    return qualifiers;
  }

  /*
   * test for fairness when "next job" doesn't match and the TM has to provide tasks from the following jobs
   */
  public void testJobHasNoTasks() throws Exception {
    final List<Integer> jobSizes = new ArrayList<Integer>();
    jobSizes.add(11); // job0
    jobSizes.add(10); // job1
    final int noOfJobs = 2;
    final Collection<Task> tasks = new ArrayList<Task>();

    // task queue:
    // [job0, ... , job0, job1, ... , job1]
    for (int i = 0; i < noOfJobs; i++) {
      for (int j = 0; j < jobSizes.get(i).intValue(); j++) {
        tasks.add(createTask("job" + i));
      }
    }

    _taskManager.addTasks(tasks);

    final List<Task> retrievedTasks = new ArrayList<Task>();
    int countJob0 = 0;
    int countJob1 = 0;
    for (int i = 0; i < tasks.size(); i++) {
      final Task task = _taskManager.getTask(TEST_WORKER, null);
      retrievedTasks.add(task);
      final String jobName = task.getProperties().get(Task.PROPERTY_JOB_NAME);
      if (jobName.equals("job0")) {
        ++countJob0;
      } else {
        ++countJob1;
      }
    }
    assertEquals(11, countJob0);
    assertEquals(10, countJob1);

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }

    // expect "round robin fairness":
    // first 20 tasks should have alternating job names
    String previousJobName = null;
    for (int i = 0; i < 20; i++) {
      final Map<String, String> taskProps = retrievedTasks.get(i).getProperties();
      final String jobName = taskProps.get(Task.PROPERTY_JOB_NAME);
      if (previousJobName != null) {
        assertFalse("Unexpected job for i=" + i + ", got two tasks for " + jobName, previousJobName.equals(jobName));
      }
      previousJobName = jobName;
    }
  }
}
