/*******************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * <pre>
 * Contributors: Jürgen Schumacher (Empolis Information Management GmbH) - implementation
 * </pre>
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test for TaskManager performance.
 * 
 * Increase timeToLive in clusterconfig.json to make this work.
 */
public class TestTaskManagerPerformance extends DeclarativeServiceTestCase {

  /** name of final worker. */
  private static final String TEST_WORKER = "TestTaskManagerPerformance";

  /** reference to task manager service. */
  private TaskManager _taskManager;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _taskManager = getService(TaskManager.class);
  }

  public void testPutAndGetTasksNonQualified1() throws Exception {
    testPutAndGetTasksNonQualified2();
  }

  public void testPutAndGetTasksNonQualified2() throws Exception {
    final int noOfTasksPerJob = 500;
    final int noOfJobs = 4;
    final Collection<Task> tasks = new ArrayList<Task>();
    for (int i = 0; i < noOfJobs; i++) {
      for (int j = 0; j < noOfTasksPerJob; j++) {
        tasks.add(createTask("job" + i));
      }
    }

    System.out.println("Adding " + tasks.size() + " tasks for " + noOfJobs + " jobs...");
    final long startAdd = System.nanoTime();
    _taskManager.addTasks(tasks);
    final double timeAddMs = (System.nanoTime() - startAdd) / 1e6;
    System.out.println("Times: complete " + timeAddMs + " ms, per task " + timeAddMs / tasks.size() + " ms");

    System.out.println("Getting " + tasks.size() + " tasks...");
    final long startGet = System.nanoTime();
    for (int i = 0; i < tasks.size(); i++) {
      _taskManager.getTask(TEST_WORKER, null);
    }
    final double timeGetMs = (System.nanoTime() - startGet) / 1e6;
    System.out.println("Times: complete " + timeGetMs + " ms, per task " + timeGetMs / tasks.size() + " ms");

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }
  }

  public void testPutAndGetTasksQualified1() throws Exception {
    testPutAndGetTasksQualified2();
  }

  public void testPutAndGetTasksQualified2() throws Exception {
    final int noOfTasksPerJob = 500;
    final int noOfJobs = 4;
    final int noOfQualifiers = 25;
    final List<String> qualifiers = new ArrayList<>(noOfQualifiers);
    for (int i = 0; i < noOfQualifiers; i++) {
      qualifiers.add(Integer.toString(i));
    }
    final List<Task> tasks = new ArrayList<Task>(noOfJobs * noOfTasksPerJob);
    for (int i = 0; i < noOfJobs; i++) {
      for (int j = 0; j < noOfTasksPerJob; j++) {
        final Task task = createTask("job" + i);
        task.setQualifier(qualifiers.get(j % noOfQualifiers));
        tasks.add(task);
      }
    }

    System.out.println("Adding " + tasks.size() + " tasks for " + noOfJobs + " jobs with " + noOfQualifiers
      + " qualifiers...");
    final long startAdd = System.nanoTime();
    _taskManager.addTasks(tasks);
    final double timeAddMs = (System.nanoTime() - startAdd) / 1e6;
    System.out.println("Times: complete " + timeAddMs + " ms, per task " + timeAddMs / tasks.size() + " ms");

    System.out.println("Getting " + tasks.size() + " tasks...");
    final long startGet = System.nanoTime();
    for (int i = 0; i < tasks.size(); i++) {
      assertNotNull("too few tasks available", _taskManager.getTask(TEST_WORKER, null, qualifiers));
    }
    final double timeGetMs = (System.nanoTime() - startGet) / 1e6;
    System.out.println("Times: complete " + timeGetMs + " ms, per task " + timeGetMs / tasks.size() + " ms");

    for (final Task task : tasks) {
      _taskManager.finishTask(TEST_WORKER, task.getTaskId(), new ResultDescription(TaskCompletionStatus.SUCCESSFUL,
        null, null, null));
    }
  }

  private Task createTask(final String jobName) {
    final Task task = new Task(UUID.randomUUID().toString(), TEST_WORKER);
    task.getProperties().put(Task.PROPERTY_JOB_NAME, jobName);
    task.getProperties().put(Task.PROPERTY_JOB_RUN_ID, "1");
    task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, "1");
    return task;
  }
}
