/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.taskmanager.test.httphandler;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.httphandler.TaskStateHandler;
import org.eclipse.smila.taskmanager.persistence.TaskStorage;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test cases for the {@link TaskStateHandler}.
 */
public class TestTaskStateHandler extends DeclarativeServiceTestCase {

  private TaskStateHandler _taskHandler;

  private TaskManager _taskManager;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    final RequestHandler service = getService(RequestHandler.class, "(uriPattern=/tasks/?$)");
    assertTrue(service instanceof TaskStateHandler);
    _taskHandler = (TaskStateHandler) service;
    _taskManager = getService(TaskManager.class);
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    clearTaskStorage();
    super.tearDown();
  }

  private void clearTaskStorage() throws Exception {
    final TaskStorage taskStorage = getService(TaskStorage.class);
    taskStorage.clear();
  }

  /**
   * tests setting and getting a config via the request handler.
   */
  public void testUpdateConfig() throws Exception {
    final String tasksUrlBase = _taskHandler.getRootContextPath() + "/tasks/";
    final Record definition = DataFactory.DEFAULT.createRecord();
    definition.getMetadata().put("maxScaleUp", 5);

    assertNull(_taskHandler.process("POST", tasksUrlBase, definition, null));

    // get the complete definition
    Record returnDetails = DataFactory.DEFAULT.createRecord();
    AnyMap retrievedConfig = (AnyMap) _taskHandler.process("GET", tasksUrlBase, returnDetails, null);

    assertTrue(retrievedConfig.containsKey("maxScaleUp"));
    assertEquals(5, retrievedConfig.getLongValue("maxScaleUp").longValue());

    definition.getMetadata().put("maxScaleUp", 7);

    assertNull(_taskHandler.process("POST", tasksUrlBase, definition, null));

    // get the complete definition
    returnDetails = DataFactory.DEFAULT.createRecord();
    retrievedConfig = (AnyMap) _taskHandler.process("GET", tasksUrlBase, returnDetails, null);

    assertTrue(retrievedConfig.containsKey("maxScaleUp"));
    assertEquals(7, retrievedConfig.getLongValue("maxScaleUp").longValue());
  }

  /**
   * test for the returnDetails parameter.
   * 
   * @throws Exception
   *           on error
   */
  public void testReturnDetails() throws Exception {
    createAndAddTasksToTaskManager();

    final String tasksUrlBase = _taskHandler.getRootContextPath() + "/tasks/";

    // returnDetails missing
    {
      final Record requestRecord = DataFactory.DEFAULT.createRecord();
      final AnyMap result = (AnyMap) _taskHandler.process("GET", tasksUrlBase, requestRecord, null);
      assertTopItems(result);
      assertTaskCountersAndMissingJobsSection(result, "worker1", 1, 0);
      assertTaskCountersAndMissingJobsSection(result, "worker2", 5, 0);
    }

    // returnDetails=false
    {
      final Record requestRecord = DataFactory.DEFAULT.createRecord();
      requestRecord.getMetadata().put("returnDetails", false);
      final AnyMap result = (AnyMap) _taskHandler.process("GET", tasksUrlBase, requestRecord, null);
      assertTopItems(result);
      assertTaskCountersAndMissingJobsSection(result, "worker1", 1, 0);
      assertTaskCountersAndMissingJobsSection(result, "worker2", 5, 0);
    }

    // returnDetails=true
    {
      final Record requestRecord = DataFactory.DEFAULT.createRecord();
      requestRecord.getMetadata().put("returnDetails", true);
      final AnyMap result = (AnyMap) _taskHandler.process("GET", tasksUrlBase, requestRecord, null);
      assertTopItems(result);
      AnyMap countsPerJobs = DataFactory.DEFAULT.createAnyMap();
      countsPerJobs.put("job1", 1);
      assertTaskCounters(result, "worker1", countsPerJobs, 1, 0);
      countsPerJobs = DataFactory.DEFAULT.createAnyMap();
      countsPerJobs.put("job1", 2);
      countsPerJobs.put("job2", 3);
      assertTaskCounters(result, "worker2", countsPerJobs, 5, 0);
    }
  }

  private void createAndAddTasksToTaskManager() throws Exception {
    final List<Task> tasks = new ArrayList<Task>();
    tasks.add(createTask("job1", "worker1"));
    tasks.add(createTask("job1", "worker2"));
    tasks.add(createTask("job1", "worker2"));
    tasks.add(createTask("job2", "worker2"));
    tasks.add(createTask("job2", "worker2"));
    tasks.add(createTask("job2", "worker2"));
    _taskManager.addTasks(tasks);
  }

  private Task createTask(final String jobName, final String workerName) {
    final Task task = new Task(UUID.randomUUID().toString(), workerName);
    task.getProperties().put(Task.PROPERTY_JOB_NAME, jobName);
    task.getProperties().put(Task.PROPERTY_JOB_RUN_ID, "1");
    task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, "1");
    return task;
  }

  private void assertTaskCountersAndMissingJobsSection(final AnyMap result, final String workerName,
    final int totalTodoCount, final int totalInProgressCount) {
    assertTrue(result.containsKey("workers"));
    final AnySeq workers = result.getSeq("workers");
    for (int i = 0; i < workers.size(); ++i) {
      final AnyMap workerInfo = workers.get(i).asMap();
      if (workerName.equals(workerInfo.getStringValue("name"))) {
        assertTrue(workerInfo.containsKey("todo"));
        final AnyMap todo = workerInfo.getMap("todo");
        assertFalse(todo.containsKey("jobs"));
        assertEquals(totalTodoCount, todo.getLongValue("count").intValue());
        assertTrue(workerInfo.containsKey("inprogress"));
        final AnyMap inProgress = workerInfo.getMap("inprogress");
        assertTrue(inProgress.containsKey("count"));
        assertEquals(totalInProgressCount, inProgress.getLongValue("count").intValue());
        return;
      }
    }
    fail("Info on worker '" + workerName + "' not found");
  }

  private void assertTaskCounters(final AnyMap result, final String workerName, final AnyMap todoCountsPerJobs,
    final int totalTodoCount, final int totalInProgressCount) {
    assertTrue(result.containsKey("workers"));
    final AnySeq workers = result.getSeq("workers");
    for (int i = 0; i < workers.size(); ++i) {
      final AnyMap workerInfo = workers.get(i).asMap();
      if (workerName.equals(workerInfo.getStringValue("name"))) {
        assertTrue(workerInfo.containsKey("todo"));
        final AnyMap todo = workerInfo.getMap("todo");
        assertTrue(todo.containsKey("jobs"));
        assertCountsPerJob(todoCountsPerJobs, todo.getMap("jobs"));
        assertTrue(todo.containsKey("count"));
        assertEquals(totalTodoCount, todo.getLongValue("count").intValue());
        assertTrue(workerInfo.containsKey("inprogress"));
        final AnyMap inProgress = workerInfo.getMap("inprogress");
        assertTrue(inProgress.containsKey("count"));
        assertEquals(totalInProgressCount, inProgress.getLongValue("count").intValue());
        return;
      }
    }
    fail("Info on worker '" + workerName + "' not found");
  }

  private void assertCountsPerJob(final AnyMap expectedCountsPerJobs, final AnyMap countsPerJob) {
    for (final Map.Entry<String, Any> entry : expectedCountsPerJobs.entrySet()) {
      final String jobName = entry.getKey();
      assertEquals(entry.getValue(), countsPerJob.get(jobName));
    }
  }

  private void assertTopItems(final AnyMap result) {
    assertTrue(result.containsKey("failsafety"));
    assertTrue(result.containsKey("taskmanager"));
    assertTrue(result.containsKey("workers"));
    assertTrue(result.containsKey("maxScaleUp"));
    assertTrue(result.containsKey("scaleUp"));
  }
}
