/**********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Schank (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.tika.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.tika.detect.Detector;
import org.apache.tika.io.TemporaryResources;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.mime.MediaType;
import org.eclipse.smila.common.mimetype.MimeTypeIdentifier;
import org.eclipse.smila.common.mimetype.MimeTypeParseException;

public class TikaMimeTypeIdentifyService implements MimeTypeIdentifier {

  private Detector _detector;

  @Override
  public String identify(final byte[] data) throws MimeTypeParseException {
    //wrap the byte[] into a stream and detect based on the stream....
    try (final ByteArrayInputStream dataStream = new ByteArrayInputStream(data);) {
      return identify(dataStream);
    } catch (final IOException e) {
      throw new MimeTypeParseException("Error while detecting mimetype.", e);
    }
  }

  @Override
  public String identify(final InputStream dataStream) throws MimeTypeParseException {
    return identify(dataStream, null);
  }

  @Override
  public String identify(final String extension) throws MimeTypeParseException {
    return identify((InputStream) null, extension);
  }

  @Override
  public String identify(final byte[] data, final String extension) throws MimeTypeParseException {
    // wrap the byte[] into a stream and detect based on the stream....
    if (data == null) {
      return identify((InputStream) null, extension);
    }
    try (final ByteArrayInputStream dataStream = new ByteArrayInputStream(data);) {
      return identify(dataStream, extension);
    } catch (final IOException e) {
      throw new MimeTypeParseException("Error while detecting mimetype.", e);
    }
  }

  @Override
  public String identify(final InputStream dataStream, final String extension) throws MimeTypeParseException {
    final Metadata metadata = new Metadata();
    final InputStream resettableInputStream;
    TemporaryResources tmp = null;
    if (dataStream == null || dataStream instanceof TikaInputStream) {
      resettableInputStream = dataStream;
    } else {
      tmp = new TemporaryResources();
      resettableInputStream = TikaInputStream.get(dataStream, tmp);
    }
    if (extension != null && !extension.isEmpty()) {
      // create a dummy file name and set the metadata
      metadata.add(Metadata.RESOURCE_NAME_KEY, "dummy." + extension);
    }
    try {
      final MediaType mt = _detector.detect(resettableInputStream, metadata);
      return mt.getBaseType().toString();
    } catch (final IOException e) {
      throw new MimeTypeParseException("Error while detecting mimetype.", e);
    } finally {
      TemporaryResourcesHelper.disposeQuietly(tmp);
    }
  }

  /**
   * @return the Detector
   */
  public Detector getDetector() {
    return _detector;
  }

  /**
   * Method for DS to set the service reference to the Detector service.
   * 
   * @param detector
   *          the Detector to set
   */
  public void setDetector(final Detector detector) {
    this._detector = detector;
  }

  /**
   * Method for DS to unset the service reference to the Detector service.
   * 
   * @param detector
   *          the Detector to unset
   */
  public void unsetDetector(final Detector detector) {
    if (this._detector == detector) {
      _detector = null;
    }
  }

}
