/**********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.test;

import org.apache.zookeeper.Transaction;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Tests for zookeeper's MultiOp functionality.
 * 
 * Since zookeeper 3.4.x it provides MultiOp transactions which allows for atomic handling of multiple update
 * operations. This seems also to be faster than using single zookeeper update operations.
 */
public class TestMultiOps extends DeclarativeServiceTestCase {

  private ZooKeeperService _zkService;

  private ZkConnection _zkCon;

  private static final String ROOT_NODE = "/TestMultiOpsDelete";

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _zkService = getService(ZooKeeperService.class);
    assertNotNull(_zkService);
    _zkCon = new ZkConnection(_zkService);
  }

  /** tests that MultiOp delete operations are faster compared to single delete operations. */
  public void testDeleteTree() throws Exception {
    final int nanosToMillis = 1000000;
    final int noOfNodes = 1000;

    // test deleteTree()
    createNodes(ROOT_NODE, noOfNodes);
    long t = System.nanoTime();
    _zkCon.deleteTreeWithoutMultiOps(ROOT_NODE);    
    long t1 = System.nanoTime() - t;
    assertNull(_zkCon.exists(ROOT_NODE));
    System.out.println("Time deleteTree(): " + (t1 / nanosToMillis));

    // test with deleteTreeMultiOps()
    createNodes(ROOT_NODE, noOfNodes);
    Transaction tx = _zkService.getClient().transaction();
    t = System.nanoTime();
    _zkCon.deleteTree(ROOT_NODE);
    tx.commit();
    long t2 = System.nanoTime() - t;
    assertNull(_zkCon.exists(ROOT_NODE));
    System.out.println("Time MultiOp: " + (t2 / nanosToMillis));

    System.out.println("MultiOp delete is " + ((t1 - t2) / nanosToMillis) + " ms faster than deleteTree()");
  }

  /** create some nodes for testing. */
  private void createNodes(final String path, final int count) throws Exception {
    _zkCon.createNode(path, ZkConnection.NO_DATA);
    for (int i = 0; i < 10; i++) {
      _zkCon.createNode(path + "/" + i, ZkConnection.NO_DATA);
      for (int j = 0; j < count / 10; j++) {
        _zkCon.createNode(path + "/" + i + "/" + j, ZkConnection.NO_DATA);
      }
    }
  }

}
