/**
 *
 */
package org.eclipse.smila.solr.search;

import java.net.URLDecoder;

import org.apache.commons.lang.NullArgumentException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest.METHOD;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.solr.SolrConfig;
import org.eclipse.smila.solr.SolrConstants.QueryLogging;
import org.eclipse.smila.solr.SolrServerService;
import org.eclipse.smila.solr.params.QueryParams;
import org.eclipse.smila.solr.params.SearchParams;
import org.eclipse.smila.solr.query.QueryTransformer;
import org.eclipse.smila.utils.service.ServiceUtils;

/**
 * @author pwissel
 *
 */
public class SolrSearchPipelet implements Pipelet {

  private final Log _log = LogFactory.getLog(getClass());

  private SolrServerService _servers;

  private AnyMap _configuration;

  /*
   * (non-Javadoc)
   *
   * @see org.eclipse.smila.processing.Pipelet#configure(org.eclipse.smila.datamodel.AnyMap)
   */
  @Override
  public void configure(AnyMap configuration) throws ProcessingException {
    _configuration = configuration;
  }

  /*
   * (non-Javadoc)
   *
   * @see org.eclipse.smila.processing.Pipelet#process(org.eclipse.smila.blackboard.Blackboard, java.lang.String[])
   */
  @Override
  public String[] process(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    final SearchParams params = new SearchParams(blackboard, _configuration);
    final boolean dropOnError =
      params.getBooleanParameter(ProcessingConstants.KEY_DROP_ON_ERROR, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    final ResultCollector results = new ResultCollector(params, _log, dropOnError);
    for (final String id : recordIds) {
      try {
        params.setCurrentRecord(id);
        final QueryParams queryParams = new QueryParams(blackboard, _configuration, id);
        final String index = queryParams.getIndexName();
        if (StringUtils.isBlank(index)) {
          throw new NullArgumentException(QueryConstants.INDEXNAME);
        }
        final SolrQuery query = new QueryTransformer(queryParams, getConfig(), index).toSolrQuery();
        // query logging
        final QueryLogging logging = params.getQueryLogging();
        Record record = blackboard.getRecord(id);
        switch (logging) {
          case SMILA:
            _log.info("SmilaQueryRecord:" + XmlSerializationUtils.serialize2string(record));
            break;
          case SOLR:
            _log.info("SolrQueryString: " + query.toString());
            break;
          case BOTH:
            _log.info("SmilaQueryRecord:" + XmlSerializationUtils.serialize2string(record));
            _log.info("SolrQueryString: " + URLDecoder.decode(query.toString(), "UTF-8"));
          default:
            break;
        }
        //
        final SolrServer server = getServer(index);
        final METHOD method = params.getMethod();
        final QueryResponse response = server.query(query, method);
        record = new ResponseParser(record, getConfig(), index).toRecord(response);
        blackboard.setRecord(record);
        results.addResult(id);
      } catch (Exception exception) {
        results.addFailedResult(id, exception);
      }
    }
    return results.getResultIds();
  }

  private SolrServer getServer(final String index) throws SolrServerException, InterruptedException {
    if (_servers == null) {
      _servers = ServiceUtils.getService(SolrServerService.class);
    }
    return _servers.getServer(index);
  }

  private SolrConfig getConfig() throws InterruptedException {
    if (_servers == null) {
      _servers = ServiceUtils.getService(SolrServerService.class);
    }
    return _servers.getConfig();
  }

}
