/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobTaskProcessor;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;

/**
 * base class for JobManager ReST call handlers. Manages service reference to jobmanager, and adds common error
 * handling.
 */
public abstract class AJobManagerHandler extends JsonRequestHandler {
  /** The reference to the JobRunDataProvider service. */
  private JobRunDataProvider _jobRunDataProvider;

  /** The reference to the JobRunEngine service. */
  private JobRunEngine _jobRunEngine;

  /** The reference to the JobTaskProcessor service. */
  private JobTaskProcessor _jobTaskProcessor;

  /** The reference to the DefinitionPersistence service. */
  private DefinitionPersistence _defPersistence;

  /** get {@link JobRunDataProvider} service. */
  public JobRunDataProvider getJobRunDataProvider() {
    return _jobRunDataProvider;
  }

  /** get {@link JobRunEngine} service. */
  public JobRunEngine getJobRunEngine() {
    return _jobRunEngine;
  }

  /** get {@link JobTaskProcessor} service. */
  public JobTaskProcessor getJobTaskProcessor() {
    return _jobTaskProcessor;
  }

  /** get {@link DefinitionPersistence} service. */
  public DefinitionPersistence getDefinitionPersistence() {
    return _defPersistence;
  }

  /**
   * Set GET instead of POST as basic method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetRequest(method);
  }

  /**
   * accept GET requests.
   */
  protected boolean isGetRequest(final String method) {
    return "GET".equals(method);
  }

  /**
   * accept POST requests.
   */
  protected boolean isPostRequest(final String method) {
    return "POST".equals(method);
  }

  /**
   * accept DELETE requests.
   */
  protected boolean isDeleteRequest(final String method) {
    return "DELETE".equals(method);
  }

  /**
   * accept GET or DELETE requests.
   */
  protected boolean isGetOrDeleteRequest(final String method) {
    return isGetRequest(method) || isDeleteRequest(method);
  }

  /**
   * accept GET or POST requests.
   */
  protected boolean isGetOrPostRequest(final String method) {
    return isGetRequest(method) || isPostRequest(method);
  }

  /**
   * Adapt error handling of
   * {@link org.eclipse.smila.http.server.json.JsonRequestHandler#getErrorStatus(String, String, Throwable)}. to
   * JobManager exceptions:
   * <ul>
   * <li> {@link IllegalArgumentException}, {@link InvalidConfigException}, {@link InvalidDefinitionException},
   * {@link InvalidValueTypeException}: 400 BAD REQUEST
   * <li> {@link PersistenceException} : 500 INTERNAL SERVER ERROR
   * </ul>
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException || ex instanceof InvalidConfigException
      || ex instanceof InvalidDefinitionException || ex instanceof InvalidValueTypeException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof NotFoundHTTPResult) {
      // check NotFoundHTTPResult before PersistenceException to handle ConfigNotFoundException correctly.
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof PersistenceException) {
      return HttpStatus.INTERNAL_SERVER_ERROR;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /** set OSGI service. */
  public void setJobRunDataProvider(final JobRunDataProvider jobRunDataProvider) {
    _jobRunDataProvider = jobRunDataProvider;
  }

  /** unset OSGI service. */
  public void unsetJobRunDataProvider(final JobRunDataProvider jobRunDataProvider) {
    if (_jobRunDataProvider == jobRunDataProvider) {
      _jobRunDataProvider = null;
    }
  }

  /** set OSGI service. */
  public void setJobRunEngine(final JobRunEngine jobRunEngine) {
    _jobRunEngine = jobRunEngine;
  }

  /** unset OSGI service. */
  public void unsetJobRunEngine(final JobRunEngine jobRunEngine) {
    if (_jobRunEngine == jobRunEngine) {
      _jobRunEngine = null;
    }
  }

  /** set OSGI service. */
  public void setJobTaskProcessor(final JobTaskProcessor jobTaskProcessor) {
    _jobTaskProcessor = jobTaskProcessor;
  }

  /** unset OSGI service. */
  public void unsetJobTaskProcessor(final JobTaskProcessor jobTaskProcessor) {
    if (_jobTaskProcessor == jobTaskProcessor) {
      _jobTaskProcessor = null;
    }
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }
}
