/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;

/**
 * Helper class for transforming valueExpressions.
 */
public final class TaskParameterUtils {

  private static final DataFactory FACTORY = DataFactory.DEFAULT;

  /**
   * Private constructor to avoid instance creation.
   */
  private TaskParameterUtils() {
  }

  /**
   * merge parameters into a single map, copying parameters prefixed with workerName over the unprefixed parameters.
   * actionParameters override workflowParameters override jobParameters. Finally evaluate all contained
   * ValueExpressions.
   */
  public static AnyMap mergeAndEvaluateParameters(final AnyMap jobParameter, final AnyMap workflowParameters,
    final AnyMap actionParameters, final String workerName) {
    final AnyMap parameters = FACTORY.createAnyMap();
    mergeParameters(parameters, jobParameter, workerName);
    mergeParameters(parameters, workflowParameters, workerName);
    mergeParameters(parameters, actionParameters, null);
    return evaluateParameters(parameters);
  }

  private static void mergeParameters(final AnyMap targetParameters, final AnyMap parametersToMerge,
    final String prefix) {
    if (parametersToMerge == null || parametersToMerge.isEmpty()) {
      return;
    }
    final String checkPrefix = prefix == null ? null : prefix + ".";
    final List<Entry<String, Any>> prefixedParamaters = new ArrayList<>(parametersToMerge.size());
    for (final Entry<String, Any> entry : parametersToMerge.entrySet()) {
      targetParameters.put(entry.getKey(), entry.getValue());
      if (checkPrefix != null && entry.getKey().startsWith(checkPrefix)) {
        prefixedParamaters.add(entry);
      }
    }
    if (!prefixedParamaters.isEmpty()) {
      for (final Entry<String, Any> entry : prefixedParamaters) {
        targetParameters.put(entry.getKey().substring(checkPrefix.length()), entry.getValue());
      }
    }
  }

  /**
   * Transforms a map with value expressions into a map with strings as value.
   * 
   * @param map
   *          The to be transformed map
   * @return The transformed map
   */
  public static AnyMap evaluateParameters(final AnyMap map) {
    if (map == null) {
      return map;
    }

    final AnyMap topLevelValues = FACTORY.createAnyMap();

    // evaluate top level entries only
    for (final Entry<String, Any> entry : map.entrySet()) {
      if (entry.getValue().isValue()) {
        topLevelValues.put(entry.getKey(), entry.getValue().asValue());
      }
    }
    return (AnyMap) evaluateParameter(map, topLevelValues);
  }

  /**
   * Transforms a map with value expressions into a map with strings as value.
   * 
   * @param param
   *          The param to be transformed
   * @return The transformed param
   */
  private static Any evaluateParameter(final Any param, final AnyMap topLevelValues) {
    if (param.isString()) {
      return FACTORY.createStringValue(new ValueExpression(param.asValue().asString()).evaluate(topLevelValues)
        .toString());
    } else if (param.isMap()) {
      final AnyMap evaluatedMap = FACTORY.createAnyMap();
      for (final Entry<String, Any> entry : param.asMap().entrySet()) {
        evaluatedMap.put(entry.getKey(), evaluateParameter(entry.getValue(), topLevelValues));
      }
      return evaluatedMap;
    } else if (param.isSeq()) {
      final AnySeq evaluatedSeq = FACTORY.createAnySeq();
      for (final Any entry : param.asSeq()) {
        evaluatedSeq.add(evaluateParameter(entry, topLevelValues));
      }
      return evaluatedSeq;
    }
    return param;
  }
}
