/*****************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation 
 *****************************************************************************************************************/
package org.eclipse.smila.processing.bpel.activities;

import org.apache.commons.lang.StringUtils;
import org.apache.ode.bpel.compiler.api.CompilationException;
import org.apache.ode.bpel.compiler.api.CompilationMessage;
import org.apache.ode.bpel.compiler.v2.CompilerContext;
import org.apache.ode.bpel.rtrep.v2.OExtensionActivity;
import org.apache.ode.bpel.rtrep.v2.OProcess;
import org.eclipse.smila.datamodel.xml.DOMRecordReader;
import org.eclipse.smila.processing.bpel.counter.PipeletPerformanceCounter;
import org.eclipse.smila.processing.bpel.util.BpelConstants;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * validates a pipelet invocation during BPEL compilation and creates a {@link PipeletHolder} for execution of the
 * invocation.
 */
public class PipeletActivityValidator {
  /** for parsing pipelet configurations. */
  private final DOMRecordReader _configParser = new DOMRecordReader();

  /**
   * utility method for creating error messages during activity registration. They must throw a
   * {@link org.apache.ode.bpel.compiler.api.CompilationException} which needs to be created with a
   * {@link CompilationMessage}. The causing exception can be added to the
   * {@link org.apache.ode.bpel.compiler.api.CompilationException} itself.
   * 
   * @param key
   *          activity key
   * @param message
   *          message describing the error.
   * @return CompilationMessage describing an error in phase=0.
   */
  public CompilationMessage createErrorCompilationMessage(final String key, final String message) {
    final CompilationMessage msg = new CompilationMessage();
    msg.severity = CompilationMessage.ERROR;
    msg.phase = 0;
    msg.code = key;
    msg.messageText = message;
    return msg;
  }

  /** validate the invokePipelet activity and create a PipeletHolder containing the details and pipelet configuration. */
  public PipeletHolder validateActivity(final OProcess pipelineProcess, final OExtensionActivity activity,
    final Element content, final String key, final CompilerContext compilerContext) {
    final PipeletHolder instance = new PipeletHolder(key);
    parsePipeletClass(instance, content);
    parseVariables(instance, content, compilerContext);
    parseConfiguration(instance, content);
    final String counterName = activity.name + "@" + activity.debugInfo.startLine;
    instance.setCounter(new PipeletPerformanceCounter(pipelineProcess.getName(), instance.getClassName(),
      counterName));
    return instance;
  }

  /** read and validate pipelet class name. */
  private void parsePipeletClass(final PipeletHolder instance, final Element content) {
    instance.setClassName(getAttributeOfElement(content, BpelConstants.TAG_PIPELET, BpelConstants.ATTR_CLASS));
    if (instance.getClassName() == null) {
      throw new CompilationException(createErrorCompilationMessage(instance.getKey(),
        "Missing definition of pipelet class"));
    }
  }

  /** read and validate variable names. */
  private void parseVariables(final PipeletHolder instance, final Element content,
    final CompilerContext compilerContext) {
    final String input = getAttributeOfElement(content, BpelConstants.TAG_VARIABLES, BpelConstants.ATTR_INPUT);
    if (input == null) {
      throw new CompilationException(createErrorCompilationMessage(instance.getKey(), "Missing input attribute"));
    }
    checkVariableExists(instance, input, compilerContext);
    final String output = getAttributeOfElement(content, BpelConstants.TAG_VARIABLES, BpelConstants.ATTR_OUTPUT);
    checkVariableExists(instance, output, compilerContext);
    final String index = getAttributeOfElement(content, BpelConstants.TAG_VARIABLES, BpelConstants.ATTR_INDEX);
    checkVariableExists(instance, index, compilerContext);
    instance.setInputVariable(input);
    instance.setOutputVariable(output);
    instance.setIndexVariable(index);
  }

  /**
   * find element with given local name (proc: namespace) and return value of given attribute. return null, if not
   * found.
   * 
   * @param content
   *          an element to search in.
   * @param localName
   *          local name of element to look for.
   * @param attribute
   *          attribute name.
   * @return attribute value if found, else null.
   */
  private String getAttributeOfElement(final Element content, final String localName, final String attribute) {
    String attributeValue = null;
    final NodeList nodes = content.getElementsByTagNameNS(BpelConstants.NAMESPACE_PROCESSOR, localName);
    if (nodes.getLength() > 0) {
      attributeValue = ((Element) nodes.item(0)).getAttribute(attribute);
      if (StringUtils.isBlank(attributeValue)) {
        attributeValue = null;
      }
    }
    return attributeValue;
  }

  /** check if variable exists. */
  private void checkVariableExists(final PipeletHolder instance, final String variable,
    final CompilerContext compilerContext) {
    if (variable != null) {
      try {
        compilerContext.resolveVariable(variable);
      } catch (final Exception ex) {
        throw new CompilationException(createErrorCompilationMessage(instance.getKey(), "Variable '" + variable
          + "' is not defined."));
      }
    }
  }

  /**
   * parse a PipeletConfiguration from the content of extension activity.
   * 
   * @param instance
   *          PipeletHolder to parse configuration for
   * @param content
   *          extension activity content.
   */
  private void parseConfiguration(final PipeletHolder instance, final Element content) {
    final NodeList config =
      content.getElementsByTagNameNS(BpelConstants.NAMESPACE_PROCESSOR, BpelConstants.TAG_CONFIGURATION);
    if (config.getLength() > 0) {
      final Element configElement = (Element) config.item(0);
      instance.setConfiguration(_configParser.parseMap(configElement));
    }
  }

}
