/*******************************************************************************
 * Copyright (c) 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;
import org.eclipse.smila.utils.service.ServiceUtils;

/**
 * Pipelet that writes record(s) to the given store by using the given object id.
 * 
 * The objectId parameter must be set in each record separately if multiple records should be processed and stored.
 * Otherwise they overwrite each other."
 */
public class StoreWriterPipelet implements Pipelet {

  public static final String PARAM_STORE = "store";

  public static final String PARAM_OBJECT_ID = "objectId";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** The pipelet configuration. */
  protected AnyMap _configuration;

  private ObjectStoreService _objectStore;

  private final IpcRecordWriter _recordWriter = new IpcRecordWriter();

  @Override
  public void configure(AnyMap configuration) throws ProcessingException {
    _configuration = configuration;
  }

  @Override
  public String[] process(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    final ObjectStoreService objectStore = getObjectStoreService();
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _configuration);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    if (recordIds != null) {
      for (final String id : recordIds) {
        try {
          paramAccessor.setCurrentRecord(id);
          final Record record = blackboard.getRecord(id);

          final String store = paramAccessor.getRequiredParameter(PARAM_STORE);
          final String objectId = paramAccessor.getRequiredParameter(PARAM_OBJECT_ID);
          final byte[] recordAsBon = _recordWriter.writeBinaryObject(record);

          objectStore.putObject(store, objectId, recordAsBon);

          resultCollector.addResult(id);
        } catch (final Exception e) {
          resultCollector.addFailedResult(id, e);
        }
      }
    }
    return resultCollector.getResultIds();
  }

  /** @return an objectstore service. */
  private synchronized ObjectStoreService getObjectStoreService() throws ProcessingException {
    if (_objectStore == null) {
      try {
        _objectStore = ServiceUtils.getService(ObjectStoreService.class);
      } catch (final Exception ex) {
        _log.warn("Error while waiting for ObjectStoreService service to come up.", ex);
      }
      if (_objectStore == null) {
        throw new ProcessingException("No ObjectStoreService service available, giving up");
      }
    }
    return _objectStore;
  }

}
