/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.scripting;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;

/**
 * Load script files and executes functions.
 *
 * Notes:
 * <ul>
 * <li>MUST be used by a single thread only.</li>
 * <li>MUST be closed after use.</li>
 * </ul>
 *
 *
 * Best usage pattern:
 *
 * <pre>
 * try (ScriptExecutor executor = scriptingEngine.getScriptExecutor()) {
 *   executor.loadScript(&quot;myScript&quot;);
 *   executor.call(&quot;myFunction&quot;, record);
 * }
 * </pre>
 */
public interface ScriptExecutor extends AutoCloseable {

  /**
   * load and execute file from scripting directory in this executor.
   * 
   * @param scriptFile
   *          Script file. Relative path some base directory, without suffix.
   * @return result result object from the script execution.
   * @throws ScriptingEngineException
   *           error loading or executing scripts.
   */
  Object loadScript(String scriptFile) throws ScriptingEngineException;

  /**
   * execute the function with the given record. The function must have been defined by previous
   * {@link #loadScript(String)} calls.
   * 
   * @param scriptFunction
   *          name of function. The function must be able to take one argument.
   * @param record
   *          record to process with the script.
   * @return result of function as a record.
   * @throws ScriptingEngineException
   *           error processing the script.
   */
  Record call(String scriptFunction, Record record) throws ScriptingEngineException;

  /**
   * execute the function with the given map object. The function must have been defined by previous
   * {@link #loadScript(String)} calls.
   * 
   * @param scriptFunction
   *          name of function. The function must be able to take one argument.
   * @param arguments
   *          arguments to process.
   * @return result of function as an AnyMap object.
   * @throws ScriptingEngineException
   *           error processing the script.
   */
  AnyMap call(String scriptFunction, AnyMap arguments) throws ScriptingEngineException;

  /**
   * 
   * @param extension
   *          the extension to install to the ScriptExecutor
   * @throws ScriptingEngineException
   */
  void install(Installable extension) throws ScriptingEngineException;

  /**
   * Interface for object that add something to the ScriptExecutor.
   */
  interface Installable {
    /**
     * Does the installation.
     * 
     * @param installTarget
     *          an engine specific object suitable to install extensions to.
     * @throws ScriptingEngineException
     *           installation failed.
     */
    void install(Object installTarget) throws ScriptingEngineException;
  }

  @Override
  void close();
}
