/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.worker;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.scripting.internal.RhinoInstallable;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.RecordOutput;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;

/**
 * A worker for executing scripts.
 *
 * The task/job parameters for the script and the function name can be overwritten in the record.
 */
public class ScriptProcessorWorker implements Worker {

  /** worker's name. */
  public static final String WORKER_NAME = "scriptProcessor";

  /** key for the script parameter. */
  public static final String KEY_SCRIPT_NAME = "script";

  /** key for the processing-function parameter. */
  public static final String KEY_PROCESS_FUNCTION = "function";

  /** key for the initialize-function parameter. */
  public static final String KEY_INIT_FUNCTION = "initializeFunction";

  /** name for "write attachments to output" parameter. */
  public static final String KEY_WRITE_ATTACHMENTS_TO_OUTPUT = "writeAttachmentsToOutput";

  /** parameter key for fail on error. */
  public static final String KEY_FAIL_ON_ERROR = "_failOnError";

  /** record attribute for (task) parameters. */
  public static final String DEFAULT_PARAMETERS_ATTRIBUTE = "_parameters";

  /** (optional) record attribute for overwriting script parameter. */
  public static final String ATTR_SCRIPT_NAME = "_script";

  /** (optional) record attribute for overwriting function parameter. */
  public static final String ATTR_FUNCTION_NAME = "_function";

  /** the workers input slot name . */
  public static final String INPUT_SLOT_NAME = "input";

  /** the workers output slot name . */
  public static final String OUTPUT_SLOT_NAME = "output";

  /** Default function name for processing records. */
  public static final String DEFAULT_PROCESS_FUNCTION = "processRecord";

  /** Default function name for initializing scripts. */
  public static final String DEFAULT_INIT_FUNCTION = "prepare";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** The reference to the scripting service. */
  private ScriptingEngine _scriptingEngine;

  @Override
  public String getName() {
    return WORKER_NAME;
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final AnyMap parameters = getTaskParameters(taskContext);
    final RecordInput recordInput = taskContext.getInputs().getAsRecordInput(INPUT_SLOT_NAME);
    final RecordOutput recordOutput = taskContext.getOutputs().getAsRecordOutput(OUTPUT_SLOT_NAME);
    final boolean success = processScript(parameters, recordInput, recordOutput, taskContext);
    if (!success) {
      throw new ProcessingException("None of the records of task " + taskContext.getTask()
        + " could be successfully processed, have a look at the log for details.");
    }
  }

  /**
   * Read parameters, initialize and execute script.
   *
   * Error handling: If script processing throws a non-recoverable exception, current record is skipped and processing
   * continues. If script processing throws a recoverable exception, processing is aborted cause task will be retried.
   * If loading or initialization of a script fails, processing is always aborted.
   */
  private boolean processScript(final AnyMap parameters, final RecordInput recordInput,
    final RecordOutput recordOutput, final TaskContext taskContext) throws Exception {
    final Map<String, ScriptExecutor> scriptExecutors = new HashMap<>();
    final String scriptName = getScriptName(parameters);
    final String functionName = getProcessFunctionName(parameters);
    final String initFunctionName = getInitFunctionName(parameters);
    boolean success = false; // to check if at least some records were processed successful
    try {
      boolean bulkFinished = false;
      while (!bulkFinished && !taskContext.isCanceled()) {
        final Record record = recordInput.getRecord();
        if (record == null) {
          bulkFinished = true;
        } else {
          setTaskParameters(record, parameters);
          final String recordScriptName = getScriptNameForRecord(record, scriptName);
          final ScriptExecutor scriptExecutor =
            getScriptExecutor(scriptExecutors, recordScriptName, initFunctionName, recordOutput, parameters,
              taskContext);
          final String recordFunctionName = getFunctionNameForRecord(record, functionName);
          if (!taskContext.isCanceled()) {
            success |= processRecord(scriptExecutor, record, recordFunctionName, recordOutput, taskContext);
          }
        }
      }
    } finally {
      for (final ScriptExecutor scriptExecutor : scriptExecutors.values()) {
        try {
          scriptExecutor.close();
        } catch (final Throwable e) {
          _log.warn("Error while closing ScriptExecutor", e);
        }
      }
    }
    return success;
  }

  private ScriptExecutor getScriptExecutor(final Map<String, ScriptExecutor> scriptExecutors,
    final String scriptName, final String initFunctionName, final RecordOutput output, final AnyMap parameters,
    final TaskContext taskContext) throws ScriptingEngineException {
    ScriptExecutor scriptExecutor = scriptExecutors.get(scriptName);
    if (scriptExecutor == null) {
      // script that must be used for this record is not loaded and initialized yet
      taskContext.addToCounter("scripts", 1);
      scriptExecutor = loadScript(scriptName, output, taskContext);
      initScript(scriptExecutor, scriptName, initFunctionName, parameters, taskContext);
      scriptExecutors.put(scriptName, scriptExecutor);
    }
    return scriptExecutor;
  }

  private void loadScriptInstallToRuntime(final RecordOutput output, final TaskContext taskContext,
    final ScriptExecutor scriptExecutor) throws ScriptingEngineException {
    scriptExecutor.install(new EmitFunction(output, taskContext));
    scriptExecutor.install(new RhinoInstallable() {
      @Override
      public void install(final Scriptable installScope) {
        ScriptableObject.putProperty(installScope, "workerLog",
          Context.javaToJS(taskContext.getLog(), installScope));
      }
    });
    scriptExecutor.install(new RhinoInstallable() {
      @Override
      public void install(final Scriptable installScope) {
        ScriptableObject.putProperty(installScope, "taskContext", Context.javaToJS(taskContext, installScope));
      }
    });
  }

  private ScriptExecutor loadScript(final String scriptName, final RecordOutput output,
    final TaskContext taskContext) throws ScriptingEngineException {
    ScriptExecutor scriptExecutor;
    final long timestamp = taskContext.getTimestamp();
    try {
      scriptExecutor = _scriptingEngine.getScriptExecutor();
      loadScriptInstallToRuntime(output, taskContext, scriptExecutor);
      scriptExecutor.loadScript(scriptName);
      return scriptExecutor;
    } finally {
      taskContext.measureTime("script.load", timestamp);
    }
  }

  /** initialize script by calling given init function with given parameters. */
  private void initScript(final ScriptExecutor scriptExecutor, final String scriptName, final String initFunction,
    final AnyMap parameters, final TaskContext taskContext) throws ScriptingEngineException {
    final long timestamp = taskContext.getTimestamp();
    try {
      scriptExecutor.call(initFunction, parameters);
    } catch (final ScriptingEngineException ex) {
      taskContext.getLog().warn("Failed to initialize script calling " + scriptName + "." + initFunction, ex);
      throw ex;
    } finally {
      taskContext.measureTime("script.prepare", timestamp);
    }
  }

  /**
   * process records. If processing throws a recoverable exception it is passed through so that the task can be retried
   * and may succeed then. Non-recoverable exceptions are catched and logged as warnings to the task log.
   *
   * @return true, if processing was successful, false if a non-recoverable exception occured.
   */
  private boolean processRecord(final ScriptExecutor scriptExecutor, final Record record,
    final String scriptFunction, final RecordOutput recordOutput, final TaskContext taskContext) throws Exception {
    final long timestamp = taskContext.getTimestamp();
    try {
      final Record result = scriptExecutor.call(scriptFunction, record);
      writeResultRecord(result, recordOutput, taskContext);
      return true;
    } catch (final ScriptingEngineException ex) {
      if (ex.isRecoverable()) {
        throw ex;
      } else {
        taskContext.getLog().warn("Failed to process record " + record.getId() + ", skipping it.", ex);
        return false;
      }
    } finally {
      taskContext.measureTime("script.process", timestamp);
    }
  }

  /** create AnyMap with task parameters. */
  private AnyMap getTaskParameters(final TaskContext taskContext) {
    return DataFactory.DEFAULT.cloneAnyMap(taskContext.getTask().getParameters());
  }

  /**
   * @return value of script to execute.
   * @throws IllegalArgumentException
   *           if parameter is not set.
   */
  private String getScriptName(final AnyMap parameters) {
    final String scriptName = parameters.getStringValue(KEY_SCRIPT_NAME);
    if (scriptName == null || scriptName.isEmpty()) {
      throw new IllegalArgumentException("Parameter '" + KEY_SCRIPT_NAME + "' is not set.");
    }
    return scriptName;
  }

  /** @return value of function name used for record processing. */
  private String getProcessFunctionName(final AnyMap parameters) {
    String function = parameters.getStringValue(KEY_PROCESS_FUNCTION);
    if (function == null || function.isEmpty()) {
      function = DEFAULT_PROCESS_FUNCTION;
    }
    return function;
  }

  /** @return value of function name used for initializing script. */
  private String getInitFunctionName(final AnyMap parameters) {
    String function = parameters.getStringValue(KEY_INIT_FUNCTION);
    if (function == null || function.isEmpty()) {
      function = DEFAULT_INIT_FUNCTION;
    }
    return function;
  }

  /** the script to execute may be overwritten via record attribute. */
  private String getScriptNameForRecord(final Record record, final String defaultScriptName) {
    final AnyMap metadata = record.getMetadata();
    final Any attributeValue = metadata.get(ATTR_SCRIPT_NAME);
    if (attributeValue != null && attributeValue.isString()) {
      return attributeValue.asValue().asString();
    }
    return defaultScriptName;
  }

  /** the function to execute may be overwritten via record attribute. */
  private String getFunctionNameForRecord(final Record record, final String defaultFunctionName) {
    final AnyMap metadata = record.getMetadata();
    final Any attributeValue = metadata.get(ATTR_FUNCTION_NAME);
    if (attributeValue != null && attributeValue.isString()) {
      return attributeValue.asValue().asString();
    }
    return defaultFunctionName;
  }

  /**
   * append the resulting records to the bulk. Errors on blackboard access are catched and logged as warnings to the
   * task log, as they are considered as record-specific non-recoverable errors. In any case, the blackboard is emptied
   * afterwards and attachments should be removed from binary storage (if used).
   *
   * @param recordOutput
   *          where to write the records. Can be null (is optional in worker description)
   */
  private void writeResultRecord(final Record record, final RecordOutput recordOutput, final TaskContext taskContext)
    throws ObjectStoreException, IOException {
    final boolean withAttachments = shouldWriteAttachments(taskContext);
    if (record != null && recordOutput != null) {
      if (!withAttachments) {
        record.removeAttachments();
      }
      recordOutput.writeRecord(record);
    }
  }

  /** write task parameters to record. */
  private void setTaskParameters(final Record record, final AnyMap parameters) {
    final AnyMap parameterMap = record.getMetadata().getMap(DEFAULT_PARAMETERS_ATTRIBUTE, true);
    for (final Map.Entry<String, Any> parameter : parameters.entrySet()) {
      parameterMap.put(parameter.getKey(), parameter.getValue());
    }
  }

  /** get parameter {@link #KEY_WRITE_ATTACHMENTS_TO_OUTPUT}, default value is true. */
  private boolean shouldWriteAttachments(final TaskContext taskContext) {
    final Boolean writeAttachments =
      taskContext.getTask().getParameters().getBooleanValue(KEY_WRITE_ATTACHMENTS_TO_OUTPUT);
    return writeAttachments == null ? true : writeAttachments;
  }

  /** set OSGI service. */
  public void setScriptingEngine(final ScriptingEngine scriptingEngine) {
    _scriptingEngine = scriptingEngine;
  }

  /** unset OSGI service. */
  public void unsetScriptingEngine(final ScriptingEngine scriptingEngine) {
    if (_scriptingEngine == scriptingEngine) {
      _scriptingEngine = null;
    }
  }

}
